package ru.yandex.cloud.client.identity.http;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.io.Resources;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockserver.junit.MockServerRule;
import org.mockserver.model.HttpRequest;
import org.mockserver.model.HttpResponse;

import ru.yandex.cloud.client.identity.Folder;
import ru.yandex.cloud.client.identity.IdentityClient;
import ru.yandex.cloud.client.identity.IdentityClientOptions;

import static com.google.common.net.MediaType.JSON_UTF_8;


/**
 * @author Oleg Baryshnikov
 */
public class SingleFolderTest {

    @Rule
    public MockServerRule mockServerRule = new MockServerRule(this);

    private HttpRequest template;
    private ExecutorService executor;
    private IdentityClient identityClient;

    @Before
    public void setUp() {
        template = HttpRequest.request("/v1/allFolders")
            .withMethod("GET");
        executor = Executors.newSingleThreadExecutor();
        identityClient = new HttpIdentityClient(
            IdentityClientOptions.forAddress("localhost", mockServerRule.getPort())
                .withHandlerExecutor(executor));
    }

    @After
    public void tearDown() throws Exception {
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
    }

    @Test
    public void empty() {
            mockServerRule.getClient()
                .when(template.clone()
                    .withQueryStringParameter("id", "empty-folder"))
                .respond(HttpResponse.response()
                    .withBody(readResource("empty.json"), JSON_UTF_8));

        Optional<Folder> folderO = identityClient.folder("empty-folder").join();

        Assert.assertTrue(folderO.isEmpty());
    }

    @Test
    public void single() {
        mockServerRule.getClient()
            .when(template.clone()
                .withQueryStringParameter("id", "single_folder"))
            .respond(HttpResponse.response()
                .withBody(readResource("folders-page-with-single-folder.json"), JSON_UTF_8));

        Optional<Folder> folderO = identityClient.folder("single_folder").join();
        Assert.assertTrue(folderO.isPresent());

        Folder folder = folderO.get();
        Folder expectedFolder = new Folder("my-cloud", "single_folder", "single-folder-name");
        Assert.assertEquals(expectedFolder, folder);
    }

    @Test
    public void tooMany() {
        boolean gotExpectedException = false;

        try {
            mockServerRule.getClient()
                .when(template.clone()
                    .withQueryStringParameter("id", "too_many_folders"))
                .respond(HttpResponse.response()
                    .withBody(readResource("folders-page1.json"), JSON_UTF_8));

            identityClient.folder("too_many_folders").join();
        } catch (Exception e) {
            gotExpectedException = e.getMessage().contains("too many folders found by id: too_many_folders");
        }

        Assert.assertTrue(gotExpectedException);
    }

    private String readResource(String name) {
        try {
            return Resources.toString(getClass().getResource(name), StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
