package ru.yandex.cloud.resourcemanager;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import io.grpc.Deadline;
import io.grpc.ManagedChannel;
import io.grpc.Status;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.channel.ChannelOption;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import yandex.cloud.priv.resourcemanager.v1.CloudServiceGrpc;
import yandex.cloud.priv.resourcemanager.v1.FolderServiceGrpc;
import yandex.cloud.priv.resourcemanager.v1.PCS;
import yandex.cloud.priv.resourcemanager.v1.PFS;

import ru.yandex.cloud.grpc.Futures;
import ru.yandex.cloud.grpc.Grpc;
import ru.yandex.grpc.utils.client.interceptors.MetricClientInterceptor;
import ru.yandex.solomon.util.host.HostUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * Service spec https://bb.yandex-team.ru/projects/CLOUD/repos/cloud-go/browse/private-api/yandex/cloud/priv/resourcemanager/v1/*.proto
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class GrpcResourceManagerClient implements ResourceManagerClient {
    private static final Logger logger = LoggerFactory.getLogger(GrpcResourceManagerClient.class);
    /**
     * https://a.yandex-team.ru/arc/trunk/arcadia/cloud/bitbucket/private-api/yandex/cloud/priv/resourcemanager/v1/folder_service.proto?rev=r8103093#L99
     */
    private static final int MAX_BATCH_SIZE = 1000;

    private final Duration requestTimeout;
    private final boolean resolveExistingOnly;
    private final FolderServiceGrpc.FolderServiceFutureStub folderServiceFutureStub;
    private final CloudServiceGrpc.CloudServiceFutureStub cloudServiceFutureStub;

    public GrpcResourceManagerClient(ResourceManagerClientOptions opts) {
        this.requestTimeout = opts.getRequestTimeout();
        this.resolveExistingOnly = opts.isResolveExistingOnly();

        ManagedChannel channel = NettyChannelBuilder.forAddress(opts.getHost(), opts.getPort())
                .userAgent(opts.getUserAgent())
                .keepAliveTime(10, TimeUnit.SECONDS)
                .keepAliveTimeout(1, TimeUnit.SECONDS)
                .keepAliveWithoutCalls(true)
                .enableRetry()
                .maxRetryAttempts(5)
                .executor(opts.getHandlerExecutor())
                .withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS, (int) opts.getConnectTimeout().toMillis())
                .intercept(new MetricClientInterceptor(HostUtils.getFqdn(), opts.getRegistry()))
                .build();

        this.folderServiceFutureStub = FolderServiceGrpc.newFutureStub(channel)
                .withCallCredentials(Grpc.authCredentials(opts.getTokenProvider()));
        this.cloudServiceFutureStub = CloudServiceGrpc.newFutureStub(channel)
                .withCallCredentials(Grpc.authCredentials(opts.getTokenProvider()));
    }

    @Override
    public CompletableFuture<List<Cloud>> allClouds() {
        return new CloudFetcher().start();
    }

    @Override
    public CompletableFuture<Optional<ResolvedFolder>> resolve(String folderId) {
        return resolveImpl(List.of(folderId))
                .thenApply(result -> result.isEmpty()
                        ? Optional.empty()
                        : Optional.of(result.get(0)));
    }

    @Override
    public CompletableFuture<List<ResolvedFolder>> resolve(Collection<String> folderIds) {
        if (folderIds.size() <= MAX_BATCH_SIZE) {
            return resolveImpl(folderIds);
        }

        List<ResolvedFolder> result = new ArrayList<>(folderIds.size());
        CompletableFuture<Void> future = completedFuture(null);
        for (List<String> batch : Lists.partition(List.copyOf(folderIds), MAX_BATCH_SIZE)) {
            future = future.thenCompose(ignore -> resolveImpl(batch)).thenAccept(result::addAll);
        }
        return future.thenApply(ignore -> result);
    }

    private CompletableFuture<List<ResolvedFolder>> resolveImpl(Collection<String> folderIds) {
        String requestId = UUID.randomUUID().toString();

        var request = PFS.ResolveFoldersRequest.newBuilder()
                .addAllFolderIds(folderIds)
                .setResolveExistingOnly(resolveExistingOnly)
                .build();

        var stub = this.folderServiceFutureStub
                .withInterceptors(Grpc.addCallMeta(requestId))
                .withDeadline(Deadline.after(requestTimeout.toMillis(), TimeUnit.MILLISECONDS));

        return Futures.whenComplete(stub.resolve(request), (response, throwable) -> {
            if (throwable != null) {
                logger.error("cannot resolve folder, requestId={}", requestId, throwable);
                throw Status.UNKNOWN
                        .withDescription("can not resolve folder, requestId=" + requestId)
                        .withCause(throwable)
                        .asRuntimeException();
            }

            if (response.getResolvedFoldersCount() == 0) {
                logger.error("no folders resolved, requestId={}", requestId);
                return List.of();
            }

            return response.getResolvedFoldersList()
                    .stream()
                    .map(folder -> new ResolvedFolder(folder.getId(), folder.getCloudId()))
                    .collect(Collectors.toList());
        });
    }

    private class CloudFetcher {

        private final List<Cloud> result = new ArrayList<>();
        private final CompletableFuture<List<Cloud>> resultFuture = new CompletableFuture<>();

        public CompletableFuture<List<Cloud>> start() {
            requestNext("");
            return resultFuture;
        }

        private void requestNext(String nextPageToken) {
            try {
                var requestId = UUID.randomUUID().toString();
                var stub = cloudServiceFutureStub
                        .withInterceptors(Grpc.addCallMeta(requestId))
                        .withDeadline(Deadline.after(requestTimeout.toMillis(), TimeUnit.MILLISECONDS));
                var request = PCS.ListCloudsRequest.newBuilder()
                        .setPageSize(MAX_BATCH_SIZE)
                        .setPageToken(nextPageToken)
                        .build();
                Futures.whenComplete(stub.list(request), (response, throwable) -> {
                    handleResponse(response, throwable, requestId);
                    return completedFuture(null);
                });
            } catch (Throwable ex) {
                resultFuture.completeExceptionally(ex);
            }
        }

        private void handleResponse(PCS.ListCloudsResponse response, @Nullable Throwable throwable, String requestId) {
            if (throwable != null) {
                logger.error("cannot list clouds, requestId={}", requestId, throwable);
                resultFuture.completeExceptionally(Status.INTERNAL
                        .withDescription("cannot list clouds, requestId=" + requestId)
                        .asRuntimeException());
            }
            response.getCloudsList().stream()
                    .map(cloud -> new Cloud(cloud.getId(), cloud.getName()))
                    .collect(Collectors.toCollection(() -> result));
            if (StringUtils.isEmpty(response.getNextPageToken())) {
                resultFuture.complete(result);
            } else if (!resultFuture.isCancelled()) {
                requestNext(response.getNextPageToken());
            }
        }
    }
}
