package ru.yandex.cloud.resourcemanager;

import java.time.Duration;
import java.util.concurrent.Executor;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.google.common.util.concurrent.MoreExecutors;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Objects.requireNonNull;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ResourceManagerClientOptions {
    private static final String HOST_PROD = "rm.private-api.cloud.yandex.net";
    private static final String HOST_PREPROD = "rm.private-api.cloud-preprod.yandex.net";

    private static final int PORT_INTERNAL = 4284;

    private final String host;
    private final int port;

    private String userAgent = "Solomon";
    private Executor handlerExecutor = MoreExecutors.directExecutor();
    private Duration connectTimeout = Duration.ofSeconds(5);
    private Duration requestTimeout = Duration.ofSeconds(10);
    private TokenProvider tokenProvider = TokenProvider.of("no-token");
    private MetricRegistry registry = MetricRegistry.root();
    private boolean resolveExistingOnly;

    private ResourceManagerClientOptions(String host, int port) {
        this.host = host;
        this.port = port;
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public Executor getHandlerExecutor() {
        return handlerExecutor;
    }

    public Duration getConnectTimeout() {
        return connectTimeout;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public TokenProvider getTokenProvider() {
        return tokenProvider;
    }

    public MetricRegistry getRegistry() {
        return registry;
    }

    public boolean isResolveExistingOnly() {
        return resolveExistingOnly;
    }

    public static ResourceManagerClientOptions forAddress(String host, int port) {
        return new ResourceManagerClientOptions(requireNonNull(host, "host"), port);
    }

    public static ResourceManagerClientOptions forProd() {
        return forAddress(HOST_PROD, PORT_INTERNAL);
    }

    public static ResourceManagerClientOptions forPreprod() {
        return forAddress(HOST_PREPROD, PORT_INTERNAL);
    }

    public ResourceManagerClientOptions withUserAgent(String userAgent) {
        this.userAgent = requireNonNull(userAgent);
        return this;
    }

    public ResourceManagerClientOptions withHandlerExecutor(@WillNotClose Executor executor) {
        this.handlerExecutor = requireNonNull(executor);
        return this;
    }

    public ResourceManagerClientOptions withConnectTimeout(Duration timeout) {
        this.connectTimeout = requireNonNull(timeout);
        return this;
    }

    public ResourceManagerClientOptions withRequestTimeout(Duration timeout) {
        this.requestTimeout = requireNonNull(timeout);
        return this;
    }

    public ResourceManagerClientOptions withTokenProvider(TokenProvider tokenProvider) {
        this.tokenProvider = requireNonNull(tokenProvider);
        return this;
    }

    public ResourceManagerClientOptions withRegistry(MetricRegistry registry) {
        this.registry = registry;
        return this;
    }

    public ResourceManagerClientOptions withResolveExistingOnly(boolean resolveExistingOnly) {
        this.resolveExistingOnly = resolveExistingOnly;
        return this;
    }
}
