package ru.yandex.cloud.session;

import java.time.Instant;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Sergey Polovko
 */
public interface Session {

    default boolean isCreated() {
        return this instanceof Created;
    }

    default Created asCreated() {
        return (Created) this;
    }

    default boolean isPresent() {
        return this instanceof Present;
    }

    default Present asPresent() {
        return (Present) this;
    }

    default boolean isAbsent() {
        return this instanceof Absent;
    }

    default Absent asAbsent() {
        return (Absent) this;
    }

    /**
     * New session was created, provided cookies must be sent to user.
     */
    @ParametersAreNonnullByDefault
    class Created implements Session {
        private final List<String> cookies;
        private final Instant expiresAt;

        public Created(List<String> cookies, Instant expiresAt) {
            this.cookies = cookies;
            this.expiresAt = expiresAt;
        }

        public List<String> getCookies() {
            return cookies;
        }

        public Instant getExpiresAt() {
            return expiresAt;
        }

        @Override
        public String toString() {
            return "Session.Created{cookies=" + cookies + ", expiresAt=" + expiresAt + '}';
        }
    }

    /**
     * Already initialized session was found.
     */
    @ParametersAreNonnullByDefault
    class Present implements Session {
        private final String accountId;
        @Nullable
        private final String login;

        public Present(String accountId, @Nullable String login) {
            this.accountId = accountId;
            this.login = login;
        }

        public String getAccountId() {
            return accountId;
        }

        public Optional<String> getLogin() {
            return Optional.ofNullable(login);
        }

        @Override
        public String toString() {
            return "Session.Present{accountId='" + accountId + ", login=" + login + "'}";
        }
    }

    /**
     * Session was not found, so user bust be redirected to provided URL to start a new session.
     */
    @ParametersAreNonnullByDefault
    class Absent implements Session {
        private final String redirectUrl;

        public Absent(String redirectUrl) {
            this.redirectUrl = redirectUrl;
        }

        public String getRedirectUrl() {
            return redirectUrl;
        }

        @Override
        public String toString() {
            return "Session.Absent{redirectUrl='" + redirectUrl + '\'' + '}';
        }
    }
}
