package ru.yandex.cloud.session;

import java.time.Duration;
import java.util.concurrent.Executor;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.google.common.util.concurrent.MoreExecutors;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Objects.requireNonNull;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class SessionClientOptions {

    private final String host;
    private final int port;

    private String oauthEndpoint = "";
    private String userAgent = "Solomon";
    private Executor handlerExecutor = MoreExecutors.directExecutor();
    private Duration connectTimeout = Duration.ofSeconds(5);
    private Duration requestTimeout = Duration.ofSeconds(10);
    private TokenProvider tokenProvider = TokenProvider.of("no-token");
    private MetricRegistry registry = MetricRegistry.root();

    private SessionClientOptions(String host, int port) {
        this.host = host;
        this.port = port;
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    public String getOauthEndpoint() {
        return oauthEndpoint;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public Executor getHandlerExecutor() {
        return handlerExecutor;
    }

    public Duration getConnectTimeout() {
        return connectTimeout;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public TokenProvider getTokenProvider() {
        return tokenProvider;
    }

    public MetricRegistry getRegistry() {
        return registry;
    }

    public static SessionClientOptions forAddress(String host, int port) {
        return new SessionClientOptions(requireNonNull(host, "host"), port);
    }

    public SessionClientOptions withOauthEndpoint(String oauthEndpoint) {
        this.oauthEndpoint = requireNonNull(oauthEndpoint);
        return this;
    }

    public SessionClientOptions withUserAgent(String userAgent) {
        this.userAgent = requireNonNull(userAgent);
        return this;
    }

    public SessionClientOptions withHandlerExecutor(@WillNotClose Executor executor) {
        this.handlerExecutor = requireNonNull(executor);
        return this;
    }

    public SessionClientOptions withConnectTimeout(Duration timeout) {
        this.connectTimeout = requireNonNull(timeout);
        return this;
    }

    public SessionClientOptions withRequestTimeout(Duration timeout) {
        this.requestTimeout = requireNonNull(timeout);
        return this;
    }

    public SessionClientOptions withTokenProvider(TokenProvider tokenProvider) {
        this.tokenProvider = requireNonNull(tokenProvider);
        return this;
    }

    public SessionClientOptions withRegistry(MetricRegistry registry) {
        this.registry = registry;
        return this;
    }
}
