package ru.yandex.cloud.session.grpc;

import java.time.Duration;
import java.time.Instant;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import io.grpc.Deadline;
import io.grpc.ManagedChannel;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.channel.ChannelOption;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import yandex.cloud.priv.oauth.v1.Claims;
import yandex.cloud.priv.oauth.v1.OAuth;
import yandex.cloud.priv.oauth.v1.SessionServiceGrpc;

import ru.yandex.cloud.grpc.Futures;
import ru.yandex.cloud.grpc.Grpc;
import ru.yandex.cloud.session.Session;
import ru.yandex.cloud.session.SessionClient;
import ru.yandex.cloud.session.SessionClientOptions;
import ru.yandex.grpc.utils.client.interceptors.MetricClientInterceptor;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * Service spec https://bb.yandex-team.ru/projects/CLOUD/repos/cloud-go/browse/private-api/yandex/cloud/priv/oauth/v1/session_service.proto
 *
 * @author Sergey Polovko
 */
public class GrpcSessionClient implements SessionClient {
    private static final Logger logger = LoggerFactory.getLogger(GrpcSessionClient.class);

    private final Duration requestTimeout;
    private final SessionServiceGrpc.SessionServiceFutureStub stub;

    public GrpcSessionClient(SessionClientOptions opts) {
        this.requestTimeout = opts.getRequestTimeout();

        ManagedChannel channel = NettyChannelBuilder.forAddress(opts.getHost(), opts.getPort())
                .userAgent(opts.getUserAgent())
                .keepAliveTime(10, TimeUnit.SECONDS)
                .keepAliveTimeout(1, TimeUnit.SECONDS)
                .keepAliveWithoutCalls(true)
                .enableRetry()
                .maxRetryAttempts(5)
                .executor(opts.getHandlerExecutor())
                .withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS, (int) opts.getConnectTimeout().toMillis())
                .intercept(new MetricClientInterceptor(HostUtils.getFqdn(), opts.getRegistry()))
                .build();

        this.stub = SessionServiceGrpc.newFutureStub(channel)
                .withCallCredentials(Grpc.authCredentials(opts.getTokenProvider()));
    }

    @Override
    public CompletableFuture<? extends Session> check(String domain, String cookieHeader, String federationId) {
        var request = OAuth.CheckSessionRequest.newBuilder()
                .setHost(domain)
                .setCookieHeader(cookieHeader)
                .setFederationId(federationId)
                .build();

        String requestId = UUID.randomUUID().toString();
        var stub = this.stub
                .withInterceptors(Grpc.addCallMeta(requestId))
                .withDeadline(Deadline.after(requestTimeout.toMillis(), TimeUnit.MILLISECONDS));

        return Futures.whenComplete(stub.check(request), (response, throwable) -> {
            if (throwable != null) {
                var authRequired = Grpc.readStatusDetails(throwable, OAuth.AuthorizationRequired.class);
                if (authRequired != null) {
                     return new Session.Absent(authRequired.getAuthorizeUrl());
                } else {
                    logger.error("cannot check session, requestId={}", requestId, throwable);
                    return null;
                }
            }

            Claims.SubjectClaims subjectClaims = response.getSubjectClaims();

            String accountId = subjectClaims.getSub();
            String login = subjectClaims.getPreferredUsername();
            if (!login.isEmpty() && login.endsWith("@yandex-team.ru")) {
                login = login.replace("@yandex-team.ru", "");
            } else {
                login = null;
            }

            return new Session.Present(accountId, login);
        });
    }

    @Override
    public CompletableFuture<? extends Session> create(String domain, String accessToken) {
        var request = OAuth.CreateSessionRequest.newBuilder()
                .setDomain(domain)
                .setAccessToken(accessToken)
                .build();

        String requestId = UUID.randomUUID().toString();
        var stub = this.stub
                .withInterceptors(Grpc.addCallMeta(requestId))
                .withDeadline(Deadline.after(requestTimeout.toMillis(), TimeUnit.MILLISECONDS));

        return Futures.whenComplete(stub.create(request), (response, throwable) -> {
            if (throwable != null) {
                var authRequired = Grpc.readStatusDetails(throwable, OAuth.AuthorizationRequired.class);
                if (authRequired != null) {
                    return new Session.Absent(authRequired.getAuthorizeUrl());
                } else {
                    logger.error("cannot create session, requestId={}", requestId, throwable);
                    return null;
                }
            }

            Instant expiresAt = Proto.toInstant(response.getExpiresAt());
            return new Session.Created(response.getSetCookieHeaderList(), expiresAt);
        });
    }
}
