package ru.yandex.cloud.token;

import java.time.Duration;
import java.util.concurrent.Executor;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.google.common.util.concurrent.MoreExecutors;

import static java.util.Objects.requireNonNull;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class IamTokenClientOptions {

    private static final String HOST_PROD = "ts.private-api.cloud.yandex.net";
    private static final String HOST_PREPROD = "ts.private-api.cloud-preprod.yandex.net";

    private static final int PORT_INTERNAL = 4282;

    private final String host;
    private final int port;

    private String userAgent = "Solomon";
    private Executor handlerExecutor = MoreExecutors.directExecutor();
    private Duration connectTimeout = Duration.ofSeconds(5);
    private Duration requestTimeout = Duration.ofSeconds(10);

    private IamTokenClientOptions(String host, int port) {
        this.host = host;
        this.port = port;
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public Executor getHandlerExecutor() {
        return handlerExecutor;
    }

    public Duration getConnectTimeout() {
        return connectTimeout;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public static IamTokenClientOptions forAddress(String host, int port) {
        return new IamTokenClientOptions(requireNonNull(host, "host"), port);
    }

    public static IamTokenClientOptions forProd() {
        return forAddress(HOST_PROD, PORT_INTERNAL);
    }

    public static IamTokenClientOptions forPreprod() {
        return forAddress(HOST_PREPROD, PORT_INTERNAL);
    }

    public IamTokenClientOptions withUserAgent(String userAgent) {
        this.userAgent = requireNonNull(userAgent);
        return this;
    }

    public IamTokenClientOptions withHandlerExecutor(@WillNotClose Executor executor) {
        this.handlerExecutor = requireNonNull(executor);
        return this;
    }

    public IamTokenClientOptions withConnectTimeout(Duration timeout) {
        this.connectTimeout = requireNonNull(timeout);
        return this;
    }

    public IamTokenClientOptions withRequestTimeout(Duration timeout) {
        this.requestTimeout = requireNonNull(timeout);
        return this;
    }
}
