package ru.yandex.cloud.token;

import java.nio.file.Path;
import java.security.PrivateKey;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;

import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

/**
 * More documentation about cloud JWT https://cloud.yandex.ru/docs/iam/operations/iam-token/create-for-sa#jwt-create
 *
 * @author Sergey Polovko
 */
public class Jwt {

    /**
     * Audience constant according to documentation.
     */
    static final String IAM_AUDIENCE = "https://iam.api.cloud.yandex.net/iam/v1/tokens";

    private Jwt() {
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    /**
     * BUILDER
     */
    public static final class Builder implements Cloneable {
        private String accountId;
        private String keyId;
        private PrivateKey privateKey;
        private Duration ttl = Duration.ofHours(1);

        public String getAccountId() {
            return accountId;
        }

        public String getKeyId() {
            return keyId;
        }

        public PrivateKey getPrivateKey() {
            return privateKey;
        }

        public Duration getTtl() {
            return ttl;
        }

        public Builder withAccountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public Builder withKeyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        public Builder withPrivateKey(PrivateKey privateKey) {
            this.privateKey = privateKey;
            return this;
        }

        public Builder withPrivateKey(Path pemFile) {
            return withPrivateKey(Pem.fromFile(pemFile));
        }

        public Builder withPrivateKey(String pemContent) {
            return withPrivateKey(Pem.fromString(pemContent));
        }

        public Builder withTtl(Duration ttl) {
            this.ttl = ttl;
            return this;
        }

        @Override
        public Builder clone() {
            try {
                return (Builder) super.clone();
            } catch (CloneNotSupportedException e) {
                throw new RuntimeException(e);
            }
        }

        public String build() {
            Instant now = Instant.now();
            return Jwts.builder()
                    .setIssuer(accountId)
                    .setHeaderParam("kid", keyId)
                    .setAudience(IAM_AUDIENCE)
                    .setIssuedAt(Date.from(now))
                    .setExpiration(Date.from(now.plusMillis(ttl.toMillis())))
                    .signWith(privateKey, SignatureAlgorithm.PS256)
                    .compact();
        }
    }
}
