package ru.yandex.cloud.token.grpc;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.protobuf.Timestamp;
import io.grpc.Deadline;
import io.grpc.ManagedChannel;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.channel.ChannelOption;
import yandex.cloud.priv.iam.v1.IamTokenServiceGrpc;
import yandex.cloud.priv.iam.v1.PITS;

import ru.yandex.cloud.token.IamToken;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.IamTokenClientOptions;
import ru.yandex.grpc.utils.client.interceptors.MetricClientInterceptor;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * Service spec https://bb.yandex-team.ru/projects/CLOUD/repos/cloud-go/browse/private-api/yandex/cloud/priv/iam/v1/iam_token_service.proto
 *
 * @author Sergey Polovko
 */
public class GrpcIamTokenClient implements IamTokenClient {

    private final Executor executor;
    private final Duration requestTimeout;
    private final IamTokenServiceGrpc.IamTokenServiceFutureStub stub;

    public GrpcIamTokenClient(IamTokenClientOptions opts) {
        this.executor = opts.getHandlerExecutor();
        this.requestTimeout = opts.getRequestTimeout();

        ManagedChannel channel = NettyChannelBuilder.forAddress(opts.getHost(), opts.getPort())
                .userAgent(opts.getUserAgent())
                .enableRetry()
                .maxRetryAttempts(5)
                .executor(executor)
                .offloadExecutor(executor)
                .withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS, (int) opts.getConnectTimeout().toMillis())
                .keepAliveTime(10, TimeUnit.SECONDS)
                .keepAliveTimeout(1, TimeUnit.SECONDS)
                .keepAliveWithoutCalls(true)
                .enableRetry()
                .intercept(new MetricClientInterceptor(HostUtils.getFqdn(), MetricRegistry.root()))
                .build();

        this.stub = IamTokenServiceGrpc.newFutureStub(channel);
    }

    @Override
    public CompletableFuture<IamToken> createByJwt(String jwt) {
        var deadline = Deadline.after(requestTimeout.toMillis(), TimeUnit.MILLISECONDS);
        var request = PITS.CreateIamTokenRequest.newBuilder()
                .setJwt(jwt)
                .build();

        var future = stub.withDeadline(deadline).create(request);
        return toCompletableFuture(future)
                .thenApply(response -> {
                    Timestamp exp = response.getExpiresAt();
                    Instant expiresAt = Instant.ofEpochSecond(exp.getSeconds(), exp.getNanos());
                    return new IamToken(response.getIamToken(), expiresAt);
                });
    }

    private <T> CompletableFuture<T> toCompletableFuture(ListenableFuture<T> future) {
        var promise = new CompletableFuture<T>();
        Futures.addCallback(future, new FutureCallback<T>() {
            @Override
            public void onSuccess(T result) {
                promise.complete(result);
            }

            @Override
            public void onFailure(@Nonnull Throwable error) {
                promise.completeExceptionally(error);
            }
        }, MoreExecutors.directExecutor());
        return promise;
    }
}
