package ru.yandex.cloud.token.http;

import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpRequest.BodyPublishers;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Base64;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Charsets;

import ru.yandex.cloud.token.IamOauthClient;
import ru.yandex.cloud.token.IamOauthClientOptions;

import static java.util.Objects.requireNonNull;

/**
 * Read more about <a href="https://openid.net/specs/openid-connect-core-1_0.html#TokenEndpoint">TokenEndpoint</a>.
 *
 * @author Sergey Polovko
 */
public class HttpIamOauthClient implements IamOauthClient {

    private static final ObjectMapper mapper = new ObjectMapper();

    private final URI endpoint;
    private final Duration requestTimeout;
    private final String authHeader;
    private final HttpClient httpClient;

    public HttpIamOauthClient(IamOauthClientOptions opts) {
        this.endpoint = URI.create(opts.getEndpoint() + "/oauth/token");
        this.requestTimeout = opts.getRequestTimeout();

        this.authHeader = basicAuth(
                requireNonNull(opts.getClientId(), "clientId"),
                requireNonNull(opts.getClientSecret(), "clientSecret"));

        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(opts.getConnectTimeout())
                .executor(opts.getHandlerExecutor())
                .build();
    }

    @Override
    public CompletableFuture<String> fetchAccessToken(String code) {
        String params =
                "grant_type=authorization_code" +
                "&code=" + URLEncoder.encode(code, Charsets.UTF_8);

        var request = HttpRequest.newBuilder(endpoint)
                .POST(BodyPublishers.ofString(params))
                .header("Authorization", authHeader)
                .header("Content-Type", "application/x-www-form-urlencoded")
                .timeout(requestTimeout)
                .build();

        return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString(Charsets.UTF_8))
                .thenApply(response -> {
                    if (response.statusCode() == 200) {
                        return parseAccessToken(response.body());
                    }

                    String message = "cannot get access token from " + endpoint +
                            ", code: " + response.statusCode() +
                            ", response: " + response.body();
                    throw new IllegalStateException(message);
                });
    }

    private static String basicAuth(String username, String password) {
        return "Basic " + Base64.getEncoder().encodeToString((username + ":" + password).getBytes(Charsets.UTF_8));
    }

    private static String parseAccessToken(String response) {
        try {
            return mapper.readValue(response, AccessTokenDto.class).accessToken;
        } catch (IOException e) {
            throw new IllegalStateException("cannot parse AccessTokenDto from " + response);
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class AccessTokenDto {
        @JsonProperty("access_token")
        String accessToken;
    }
}
