package ru.yandex.cloud.token;

import java.io.StringReader;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.spec.X509EncodedKeySpec;
import java.util.Date;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;
import org.junit.Assert;
import org.junit.Test;

import static java.util.Objects.requireNonNull;
import static ru.yandex.cloud.token.IamTokenClientTest.ACCOUNT_ID;
import static ru.yandex.cloud.token.IamTokenClientTest.KEY_ID;
import static ru.yandex.cloud.token.IamTokenClientTest.PRIVATE_KEY;
import static ru.yandex.cloud.token.IamTokenClientTest.PUBLIC_KEY;

/**
 * @author Sergey Polovko
 */
public class JwtTest {

    @Test
    public void create() throws Exception {
        String jwt = Jwt.newBuilder()
                .withAccountId(ACCOUNT_ID)
                .withKeyId(KEY_ID)
                .withPrivateKey(PRIVATE_KEY)
                .build();

        var token = Jwts.parser()
                .setSigningKey(readPublicKey(PUBLIC_KEY))
                .parse(jwt);

        // header
        Assert.assertEquals(KEY_ID, token.getHeader().get("kid"));
        Assert.assertEquals("PS256", token.getHeader().get("alg"));

        // body
        Claims body = (Claims) token.getBody();
        Assert.assertEquals(ACCOUNT_ID, body.getIssuer());
        Assert.assertEquals(Jwt.IAM_AUDIENCE, body.getAudience());

        Date now = new Date();
        Assert.assertTrue(body.getIssuedAt().before(now) || body.getIssuedAt().equals(now));
        Assert.assertTrue(body.getExpiration().after(now));
    }

    private static PublicKey readPublicKey(String data) throws Exception {
        try (PemReader reader = new PemReader(new StringReader(data))) {
            PemObject mepObject = requireNonNull(reader.readPemObject(), "mepObject");
            KeyFactory keyFactory = requireNonNull(KeyFactory.getInstance("RSA"), "keyFactory");
            return keyFactory.generatePublic(new X509EncodedKeySpec(mepObject.getContent()));
        }
    }
}
