package ru.yandex.idm;

import java.time.Duration;
import java.util.concurrent.Executor;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Objects.requireNonNull;

/**
 * @author Alexey Trushkin
 */
public class IdmClientOptions {
    private final String url;
    private final String host;
    private final String protocol;
    private final String oauthToken;
    private final MetricRegistry metricRegistry;
    private final Duration connectionTimeout;
    private final Executor executor;

    private IdmClientOptions(Builder builder) {
        this.url = builder.url != null
                ? builder.url
                : "https://idm-api.yandex-team.ru";
        String[] split = this.url.split("://");
        host = split[1];
        protocol = split[0];
        this.oauthToken = requireNonNull(builder.oauthToken);
        this.metricRegistry = builder.metricRegistry != null
                ? builder.metricRegistry
                : MetricRegistry.root();
        this.executor = requireNonNull(builder.executor);
        this.connectionTimeout = builder.connectionTimeout != null
                ? builder.connectionTimeout
                : Duration.ofSeconds(10);
    }

    public Duration getConnectionTimeout() {
        return connectionTimeout;
    }

    public Executor getExecutor() {
        return executor;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getUrl() {
        return url;
    }

    public String getOauthToken() {
        return oauthToken;
    }

    public MetricRegistry getMetricRegistry() {
        return metricRegistry;
    }

    public String getHost() {
        return host;
    }

    public String getProtocol() {
        return protocol;
    }

    public static class Builder {
        @Nullable
        private String url;
        @Nullable
        private String oauthToken;
        @Nullable
        private MetricRegistry metricRegistry;
        @Nullable
        private Duration connectionTimeout;
        private Executor executor;

        private Builder() {
        }

        public Builder setConnectionTimeout(Duration duration) {
            this.connectionTimeout = duration;
            return this;
        }

        public Builder setExecutor(Executor executor) {
            this.executor = executor;
            return this;
        }

        public Builder setUrl(String url) {
            this.url = "".equals(url) ? null : url;
            return this;
        }

        public Builder setOauthToken(String oauthToken) {
            this.oauthToken = "".equals(oauthToken) ? null : oauthToken;
            return this;
        }

        public Builder setMetricRegistry(MetricRegistry metricRegistry) {
            this.metricRegistry = metricRegistry;
            return this;
        }

        public IdmClientOptions build() {
            return new IdmClientOptions(this);
        }
    }
}
