package ru.yandex.idm.http;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpStatusClass;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.ResponseStatus;

import ru.yandex.bolts.function.Function;
import ru.yandex.idm.IdmClient;
import ru.yandex.idm.IdmClientOptions;
import ru.yandex.idm.dto.ErrorResponse;
import ru.yandex.idm.dto.GetRolesResponse;
import ru.yandex.idm.dto.RoleRequestDto;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * TODO make common client
 *
 * @author Alexey Trushkin
 */
public class HttpIdmClient implements IdmClient {

    private static final Logger logger = LoggerFactory.getLogger(HttpIdmClient.class);
    private static final String X_SYSTEM_REQUEST_ID_HEADER = "X-System-Request-Id";
    private static final Duration DEFAULT_REQUEST_TIMEOUT_MILLIS = Duration.ofSeconds(30);
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withExceptionFilter(throwable -> !(throwable.getCause() instanceof ClientError) && !(throwable.getCause() instanceof FiredError))
            .withNumRetries(5)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final HttpClient httpClient;
    private final IdmClientOptions opts;

    private final ConcurrentMap<String, InnerMetrics> metricsMap = new ConcurrentHashMap<>();
    private final ObjectMapper mapper = new ObjectMapper();

    public HttpIdmClient(IdmClientOptions opts) {
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(opts.getConnectionTimeout())
                .executor(opts.getExecutor())
                .build();
        this.opts = opts;
    }

    @Override
    public CompletableFuture<Void> requestRole(RoleRequestDto requestDto) {
        CompletableFuture<Void> resultFuture = new CompletableFuture<>();
        try {
            RetryCompletableFuture.runWithRetries(() -> requestRoleInner(requestDto, resultFuture), RETRY_CONFIG);
        } catch (Throwable t) {
            resultFuture.completeExceptionally(t);
        }
        return resultFuture;
    }

    @Override
    public CompletableFuture<Void> deleteRole(RoleRequestDto requestDto) {
        UUID uuid = UUID.randomUUID();
        return CompletableFutures.safeCall(() -> RetryCompletableFuture.runWithRetries(() -> findRoleIdsInner(uuid, requestDto)
                .thenCompose(response -> {
                    var roles = response.objects.stream()
                            .filter(role -> {
                                if (requestDto.user != null) {
                                    return role.group == null;
                                } else {
                                    return role.group != null && role.group.id.equals(requestDto.group);
                                }
                            })
                            .collect(Collectors.toList());
                    if (roles.size() > 1) {
                        return CompletableFuture.failedFuture(
                                new IllegalStateException("Find request for " + requestDto + " must return one or zero role, but returned " + response +
                                        " X_SYSTEM_REQUEST_ID_HEADER=" + uuid));
                    }
                    if (roles.isEmpty()) {
                        return CompletableFuture.completedFuture(null);
                    }
                    logger.info("Idm role found:\n{}={}\n{}", X_SYSTEM_REQUEST_ID_HEADER, uuid, roles.get(0));
                    return deleteRoleInner(roles.get(0).id, requestDto);
                }), RETRY_CONFIG));
    }

    private CompletableFuture<GetRolesResponse> findRoleIdsInner(UUID uuid, RoleRequestDto requestDto) {
        String endpoint = "/api/v1/roles/";
        Map<String, String> params = new HashMap<>();
        try {
            params.put("system", requestDto.system);
            params.put("type", "active");
            if (requestDto.path != null) {
                params.put("path", requestDto.path);
            }
            params.put("fields_data", mapper.writeValueAsString(requestDto.fields));
            if (requestDto.user != null) {
                params.put("user", requestDto.user);
            } else if (requestDto.group != null) {
                params.put("group", requestDto.group.toString());
            }
        } catch (JsonProcessingException e) {
            return CompletableFuture.failedFuture(new RuntimeException("Can't serialize model", e));
        }
        URI uri = buildUri(opts, endpoint, params);

        var request = HttpRequest.newBuilder(uri)
                .GET()
                .header(HttpHeaderNames.AUTHORIZATION.toString(), "OAuth " + opts.getOauthToken())
                .header(X_SYSTEM_REQUEST_ID_HEADER, uuid.toString())
                .timeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .build();
        logger.info("Idm role try find:\n{}={}\n{}", X_SYSTEM_REQUEST_ID_HEADER, uuid, request);
        return wrapMetrics(endpoint, metrics -> httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(response -> {
                    handleException(metrics, response, uuid, requestDto);
                    try {
                        return mapper.readValue(response.body(), GetRolesResponse.class);
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }));
    }

    @Override
    public CompletableFuture<Void> deleteRoles(RoleRequestDto projectRequest) {
        return CompletableFutures.safeCall(() -> RetryCompletableFuture.runWithRetries(() -> findRoleIdsInner(UUID.randomUUID(), projectRequest)
                .thenCompose(response -> {
                    if (response.objects.isEmpty()) {
                        return CompletableFuture.completedFuture(null);
                    }
                    List<CompletableFuture<Void>> futures = new ArrayList<>(response.objects.size());
                    for (GetRolesResponse.Role object : response.objects) {
                        futures.add(deleteRoleInner(object.id, projectRequest));
                    }
                    return CompletableFutures.allOfVoid(futures);
                }), RETRY_CONFIG));
    }

    private CompletableFuture<Void> deleteRoleInner(Integer id, RoleRequestDto requestDto) {
        UUID uuid = UUID.randomUUID();
        String endpoint = "/api/v1/roles/";
        String uri = opts.getUrl() + endpoint + id + "/";
        var request = HttpRequest.newBuilder(URI.create(uri))
                .DELETE()
                .header(HttpHeaderNames.AUTHORIZATION.toString(), "OAuth " + opts.getOauthToken())
                .header(X_SYSTEM_REQUEST_ID_HEADER, uuid.toString())
                .timeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .build();
        logger.info("Idm role delete:\n{}={}\n{}", X_SYSTEM_REQUEST_ID_HEADER, uuid, request);
        return wrapMetrics(endpoint, metrics -> httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenAccept(response -> handleException(metrics, response, uuid, requestDto)));
    }

    private CompletableFuture<?> requestRoleInner(RoleRequestDto requestDto, CompletableFuture<Void> resultFuture) {
        UUID uuid = UUID.randomUUID();
        String endpoint = "/api/v1/rolerequests/";
        String uri = opts.getUrl() + endpoint;
        String body;
        try {
            body = mapper.writeValueAsString(requestDto);
        } catch (JsonProcessingException e) {
            return CompletableFuture.failedFuture(new RuntimeException("Can't serialize model", e));
        }
        var request = HttpRequest.newBuilder(URI.create(uri))
                .POST(HttpRequest.BodyPublishers.ofString(body))
                .header(HttpHeaderNames.AUTHORIZATION.toString(), "OAuth " + opts.getOauthToken())
                .header(X_SYSTEM_REQUEST_ID_HEADER, uuid.toString())
                .timeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .build();
        logger.info("Idm role request:\n{}={}\n{}\n{}", X_SYSTEM_REQUEST_ID_HEADER, uuid, request, body);
        var future = wrapMetrics(endpoint, metrics -> httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenAccept(response -> handleException(metrics, response, uuid, requestDto)));
        CompletableFutures.whenComplete(future, resultFuture);
        return future;
    }

    private <R> CompletableFuture<R> wrapMetrics(String endpoint, Function<InnerMetrics, CompletableFuture<R>> func) {
        InnerMetrics metrics = metricsMap.computeIfAbsent(endpoint, s -> new InnerMetrics(endpoint));
        var future = func.apply(metrics);
        metrics.asyncMetrics.forFuture(future);
        return future;
    }

    private void handleException(InnerMetrics metrics, HttpResponse<String> response, UUID headerUuid, RoleRequestDto request) {
        metrics.status(response.statusCode());
        if (!HttpStatusClass.SUCCESS.contains(response.statusCode())) {
            logger.error("Idm response (status {}) ({} {}):\n{}, body {}",
                    response.statusCode(), X_SYSTEM_REQUEST_ID_HEADER, headerUuid, response, response.body());
            if (response.statusCode() == 409 && response.body().contains(" уже есть такая роль ")) {
                // skip only idempotent conflict call
                return;
            }
            if (HttpStatusClass.CLIENT_ERROR.contains(response.statusCode())) {
                var message = prepareErrorMessage(response.body(), request);
                if (message.endsWith(" уволен")) {
                    throw new FiredError(message);
                }
                throw new ClientError(message);
            }
            //retry errors
            throw new IllegalStateException("Idm response status " + response.statusCode());
        }
    }

    private String prepareErrorMessage(String body, RoleRequestDto request) {
        try {
            var response = mapper.readValue(body, ErrorResponse.class);
            var message = response.getMessage();
            if (request.tvm) {
                return message.replaceAll("Пользователь", "Tvm").replaceAll("пользователь", "tvm");
            }
            return message;
        } catch (IOException ioException) {
            logger.error("Can't parse model", ioException);
        }
        return "Idm response: " + body;
    }

    @Override
    public void close() {
    }

    private URI buildUri(IdmClientOptions opts, String path, Map<String, String> params) {
        var uriBuilder = new UriBuilder()
                .setScheme(opts.getProtocol())
                .setHost(opts.getHost())
                .appendPath(path);
        params.forEach(uriBuilder::addParam);

        URI uri;
        try {
            uri = uriBuilder.build();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return uri;
    }

    private class InnerMetrics {
        private final ConcurrentMap<Integer, Rate> statusCodes = new ConcurrentHashMap<>();
        private final AsyncMetrics asyncMetrics;
        private final MetricRegistry registry;
        private final Labels commonLabels;

        private InnerMetrics(String endpoint) {
            registry = opts.getMetricRegistry();
            commonLabels = Labels.of("endpoint", endpoint);
            this.asyncMetrics = new AsyncMetrics(opts.getMetricRegistry(),
                    "idm.client.request", commonLabels);
        }

        public void status(int statusCode) {
            Labels labels = commonLabels.toBuilder()
                    .add("code", Integer.toString(statusCode))
                    .build();
            statusCodes.computeIfAbsent(statusCode,
                    code -> registry.rate("idm.client.request.status", labels)).inc();
        }
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public static class ClientError extends RuntimeException {
        public ClientError(String message) {
            super(message);
        }
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public static class FiredError extends RuntimeException {
        public FiredError(String message) {
            super(message);
        }
    }
}
