package ru.yandex.jns.config;

import java.time.Duration;
import java.util.concurrent.Executor;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Objects.requireNonNull;

/**
 * @author Alexey Trushkin
 */
public class JnsClientOptions {
    private final String url;
    private final MetricRegistry metricRegistry;
    private final Duration connectionTimeout;
    private final Executor executor;
    private final Supplier<String> tokenProvider;
    private final Supplier<String> tokenHeaderProvider;

    private JnsClientOptions(Builder builder) {
        this.url = builder.url != null
                ? builder.url
                : "https://jns.yandex-team.ru";
        this.metricRegistry = builder.metricRegistry != null
                ? builder.metricRegistry
                : MetricRegistry.root();
        this.executor = requireNonNull(builder.executor);
        this.connectionTimeout = builder.connectionTimeout != null
                ? builder.connectionTimeout
                : Duration.ofSeconds(10);
        this.tokenProvider = builder.tokenProvider;
        this.tokenHeaderProvider = builder.tokenHeaderProvider;
    }

    public Duration getConnectionTimeout() {
        return connectionTimeout;
    }

    public Executor getExecutor() {
        return executor;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getUrl() {
        return url;
    }

    public MetricRegistry getMetricRegistry() {
        return metricRegistry;
    }

    public Supplier<String> getTokenProvider() {
        return tokenProvider;
    }

    public Supplier<String> getTokenHeaderProvider() {
        return tokenHeaderProvider;
    }

    public static class Builder {
        @Nullable
        private String url;
        @Nullable
        private Supplier<String> tokenProvider;
        @Nullable
        private Supplier<String> tokenHeaderProvider;
        @Nullable
        private MetricRegistry metricRegistry;
        @Nullable
        private Duration connectionTimeout;
        private Executor executor;

        private Builder() {
        }

        public Builder setTokenProvider(@Nullable Supplier<String> tokenProvider) {
            this.tokenProvider = tokenProvider;
            return this;
        }

        public Builder setTokenHeaderProvider(@Nullable Supplier<String> tokenHeaderProvider) {
            this.tokenHeaderProvider = tokenHeaderProvider;
            return this;
        }

        public Builder setConnectionTimeout(Duration duration) {
            this.connectionTimeout = duration;
            return this;
        }

        public Builder setExecutor(Executor executor) {
            this.executor = executor;
            return this;
        }

        public Builder setUrl(String url) {
            this.url = "".equals(url) ? null : url;
            return this;
        }

        public Builder setMetricRegistry(MetricRegistry metricRegistry) {
            this.metricRegistry = metricRegistry;
            return this;
        }

        public JnsClientOptions build() {
            return new JnsClientOptions(this);
        }
    }
}

