package ru.yandex.juggler.client;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.handler.codec.http.HttpStatusClass;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.ResponseStatus;

import ru.yandex.juggler.config.JugglerClientOptions;
import ru.yandex.juggler.dto.ProjectDeleteRequestDto;
import ru.yandex.juggler.dto.ProjectDto;
import ru.yandex.juggler.dto.ProjectFindRequestDto;
import ru.yandex.juggler.dto.ProjectListDto;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.future.RetryConfig;

import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class HttpJugglerControlPlaneClient implements JugglerControlPlaneClient {

    private static final Logger logger = LoggerFactory.getLogger(HttpJugglerControlPlaneClient.class);
    private static final Duration DEFAULT_REQUEST_TIMEOUT_MILLIS = Duration.ofSeconds(30);
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withExceptionFilter(throwable -> !(throwable.getCause() instanceof ClientError))
            .withNumRetries(2)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final HttpClient httpClient;
    private final JugglerClientOptions opts;

    private final ConcurrentMap<String, InnerMetrics> metricsMap = new ConcurrentHashMap<>();
    private final ObjectMapper mapper = new ObjectMapper();

    public HttpJugglerControlPlaneClient(JugglerClientOptions opts) {
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(opts.getConnectionTimeout())
                .executor(opts.getExecutor())
                .build();
        this.opts = opts;
    }

    @Override
    public CompletableFuture<Void> deleteProject(ProjectDeleteRequestDto requestDto) {
        return executeWithRetries(Void.class, requestDto, "/v2/namespaces/remove_namespace");
    }

    @Override
    public CompletableFuture<Void> deleteProject(ProjectDto requestDto) {
        ProjectFindRequestDto dto = new ProjectFindRequestDto();
        dto.abcService = requestDto.abcService;
        dto.name = requestDto.name;
        dto.limit = 2;
        return getProject(dto)
                .thenApply(projectDto -> projectDto.orElseThrow(() -> new NotFound("can't find project for:\n " + requestDto)))
                .thenCompose(projectDto -> {
                    ProjectDeleteRequestDto deleteRequestDto = new ProjectDeleteRequestDto();
                    deleteRequestDto.id = projectDto.id;
                    return deleteProject(deleteRequestDto);
                });
    }

    @Override
    public CompletableFuture<Optional<ProjectDto>> getProject(ProjectFindRequestDto requestDto) {
        return executeWithRetries(ProjectListDto.class, requestDto, "/v2/namespaces/get_namespaces")
                .thenApply(this::getSingleOrThrow);
    }

    @Override
    public CompletableFuture<Void> updateProject(ProjectDto requestDto) {
        if (!StringUtils.isEmpty(requestDto.id)) {
            return executeWithRetries(Void.class, requestDto, "/v2/namespaces/set_namespace");
        }
        ProjectFindRequestDto dto = new ProjectFindRequestDto();
        dto.abcService = requestDto.abcService;
        dto.name = requestDto.name;
        dto.limit = 2;
        return getProject(dto)
                .thenApply(projectDto -> projectDto.orElseThrow(() -> new NotFound("can't find project for:\n " + requestDto)))
                .thenCompose(projectDto -> {
                    requestDto.id = projectDto.id;
                    return executeWithRetries(Void.class, requestDto, "/v2/namespaces/set_namespace");
                });
    }

    @Override
    public CompletableFuture<Void> createProject(ProjectDto requestDto) {
        return executeWithRetries(Void.class, requestDto, "/v2/namespaces/set_namespace");
    }

    private <T> CompletableFuture<T> request(Class<T> clz, Object requestDto, CompletableFuture<T> resultFuture, String endpoint) {
        UUID uuid = UUID.randomUUID();
        String uri = opts.getUrl() + endpoint;
        String body;
        try {
            body = mapper.writeValueAsString(requestDto);
        } catch (JsonProcessingException e) {
            return CompletableFuture.failedFuture(new RuntimeException("Can't serialize model", e));
        }
        var request = HttpRequest.newBuilder(URI.create(uri))
                .POST(HttpRequest.BodyPublishers.ofString(body))
                .header(opts.getTokenHeaderProvider().get(), opts.getTokenProvider().get())
                .timeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .build();
        logger.info("Project request:\n{}\n{}\n{}", uuid, request, body);
        var future = wrapMetrics(endpoint, metrics -> httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(response -> {
                    handleException(metrics, response, uuid);
                    if (clz == Void.class) {
                        return null;
                    }
                    try {
                        return mapper.readValue(response.body(), clz);
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }));
        CompletableFutures.whenComplete(future, resultFuture);
        return future;
    }

    private <R> CompletableFuture<R> wrapMetrics(String endpoint, Function<InnerMetrics, CompletableFuture<R>> func) {
        InnerMetrics metrics = metricsMap.computeIfAbsent(endpoint, s -> new InnerMetrics(endpoint));
        var future = func.apply(metrics);
        metrics.asyncMetrics.forFuture(future);
        return future;
    }

    private void handleException(InnerMetrics metrics, HttpResponse<String> response, UUID headerUuid) {
        metrics.status(response.statusCode());
        if (!HttpStatusClass.SUCCESS.contains(response.statusCode())) {
            logger.error("Project response (status {}) ({}):\n{}, body {}",
                    response.statusCode(), headerUuid, response, response.body());
            if (HttpStatusClass.CLIENT_ERROR.contains(response.statusCode()) && response.statusCode() != HttpStatus.TOO_MANY_REQUESTS.value()) {
                // wouldnt retry
                throw new ClientError(response.body());
            }
            //retry errors
            throw new IllegalStateException("Project response status " + response.statusCode());
        }
    }

    private <T> CompletableFuture<T> executeWithRetries(Class<T> clzResult, Object requestDto, String endpoint) {
        CompletableFuture<T> resultFuture = new CompletableFuture<>();
        try {
            runWithRetries(() -> request(clzResult, requestDto, resultFuture, endpoint), RETRY_CONFIG);
        } catch (Throwable t) {
            resultFuture.completeExceptionally(t);
        }
        return resultFuture;
    }

    private Optional<ProjectDto> getSingleOrThrow(ProjectListDto projectListDto) {
        if (projectListDto.items == null || projectListDto.items.isEmpty()) {
            return Optional.empty();
        }
        if (projectListDto.items.size() > 1) {
            throw new RuntimeException("multiple projects found:\n " + projectListDto);
        }
        return Optional.of(projectListDto.items.get(0));
    }

    @Override
    public void close() {
    }

    private class InnerMetrics {
        private final ConcurrentMap<Integer, Rate> statusCodes = new ConcurrentHashMap<>();
        private final AsyncMetrics asyncMetrics;
        private final MetricRegistry registry;
        private final Labels commonLabels;

        private InnerMetrics(String endpoint) {
            registry = opts.getMetricRegistry();
            commonLabels = Labels.of("endpoint", endpoint);
            this.asyncMetrics = new AsyncMetrics(opts.getMetricRegistry(),
                    "juggler.client.control_plane.request", commonLabels);
        }

        public void status(int statusCode) {
            Labels labels = commonLabels.toBuilder()
                    .add("code", Integer.toString(statusCode))
                    .build();
            statusCodes.computeIfAbsent(statusCode,
                    code -> registry.rate("juggler.client.control_plane.request.status", labels)).inc();
        }
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    private static class ClientError extends RuntimeException {
        public ClientError(String message) {
            super(message);
        }
    }

    @ResponseStatus(HttpStatus.NOT_FOUND)
    private static class NotFound extends RuntimeException {
        public NotFound(String message) {
            super(message);
        }
    }

}
