package ru.yandex.juggler.client;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.handler.codec.http.HttpStatusClass;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.ResponseStatus;

import ru.yandex.juggler.config.JugglerClientOptions;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.future.RetryConfig;

import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class HttpJugglerProjectChangesClient implements JugglerProjectChangesClient {

    private static final Logger logger = LoggerFactory.getLogger(HttpJugglerProjectChangesClient.class);
    private static final Duration DEFAULT_REQUEST_TIMEOUT_MILLIS = Duration.ofSeconds(30);
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withExceptionFilter(throwable -> !(throwable.getCause() instanceof ClientError))
            .withNumRetries(2)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final HttpClient httpClient;
    private final JugglerClientOptions opts;

    private final ConcurrentMap<String, InnerMetrics> metricsMap = new ConcurrentHashMap<>();
    private final ObjectMapper mapper = new ObjectMapper();

    public HttpJugglerProjectChangesClient(JugglerClientOptions opts) {
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(opts.getConnectionTimeout())
                .executor(opts.getExecutor())
                .build();
        this.opts = opts;
    }

    @Override
    public CompletableFuture<Void> deleteProject(String jsonProto) {
        return executeWithRetries(Void.class, jsonProto, "/v2/projects/push_delete");
    }

    @Override
    public CompletableFuture<Void> updateProject(String jsonProto) {
        return executeWithRetries(Void.class, jsonProto, "/v2/projects/push_update");
    }

    @Override
    public CompletableFuture<Void> createProject(String jsonProto) {
        return executeWithRetries(Void.class, jsonProto, "/v2/projects/push_create");
    }

    private <T> CompletableFuture<T> request(Class<T> clz, String jsonProto, CompletableFuture<T> resultFuture, String endpoint) {
        UUID uuid = UUID.randomUUID();
        String uri = opts.getUrl() + endpoint;
        var request = HttpRequest.newBuilder(URI.create(uri))
                .POST(HttpRequest.BodyPublishers.ofString(jsonProto))
                .header(opts.getTokenHeaderProvider().get(), opts.getTokenProvider().get())
                .header("Content-Type", "application/json")
                .timeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .build();
        logger.info("Project request:\n{}\n{}\n{}", uuid, request, jsonProto);
        var future = wrapMetrics(endpoint, metrics -> httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(response -> {
                    handleException(metrics, response, uuid);
                    if (clz == Void.class) {
                        return null;
                    }
                    try {
                        return mapper.readValue(response.body(), clz);
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }));
        CompletableFutures.whenComplete(future, resultFuture);
        return future;
    }

    private <R> CompletableFuture<R> wrapMetrics(String endpoint, Function<InnerMetrics, CompletableFuture<R>> func) {
        InnerMetrics metrics = metricsMap.computeIfAbsent(endpoint, s -> new InnerMetrics(endpoint));
        var future = func.apply(metrics);
        metrics.asyncMetrics.forFuture(future);
        return future;
    }

    private void handleException(InnerMetrics metrics, HttpResponse<String> response, UUID headerUuid) {
        metrics.status(response.statusCode());
        if (!HttpStatusClass.SUCCESS.contains(response.statusCode())) {
            logger.error("Project response (status {}) ({}):\n{}, body {}",
                    response.statusCode(), headerUuid, response, response.body());
            if (HttpStatusClass.CLIENT_ERROR.contains(response.statusCode()) && response.statusCode() != HttpStatus.TOO_MANY_REQUESTS.value()) {
                // wouldnt retry
                throw new ClientError(response.body());
            }
            //retry errors
            throw new IllegalStateException("Project response status " + response.statusCode());
        }
    }

    private <T> CompletableFuture<T> executeWithRetries(Class<T> clzResult, String jsonProto, String endpoint) {
        CompletableFuture<T> resultFuture = new CompletableFuture<>();
        try {
            runWithRetries(() -> request(clzResult, jsonProto, resultFuture, endpoint), RETRY_CONFIG);
        } catch (Throwable t) {
            resultFuture.completeExceptionally(t);
        }
        return resultFuture;
    }

    @Override
    public void close() {
    }

    private class InnerMetrics {
        private final ConcurrentMap<Integer, Rate> statusCodes = new ConcurrentHashMap<>();
        private final AsyncMetrics asyncMetrics;
        private final MetricRegistry registry;
        private final Labels commonLabels;

        private InnerMetrics(String endpoint) {
            registry = opts.getMetricRegistry();
            commonLabels = Labels.of("endpoint", endpoint);
            this.asyncMetrics = new AsyncMetrics(opts.getMetricRegistry(),
                    "juggler.project_changes_client.control_plane.request", commonLabels);
        }

        public void status(int statusCode) {
            Labels labels = commonLabels.toBuilder()
                    .add("code", Integer.toString(statusCode))
                    .build();
            statusCodes.computeIfAbsent(statusCode,
                    code -> registry.rate("juggler.project_changes_client.control_plane.request.status", labels)).inc();
        }
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    private static class ClientError extends RuntimeException {
        public ClientError(String message) {
            super(message);
        }
    }
}
