package ru.yandex.juggler.client;

import java.net.http.HttpClient;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.juggler.dto.EventStatus;
import ru.yandex.juggler.dto.GetConfigResponse;
import ru.yandex.juggler.dto.JugglerEvent;
import ru.yandex.juggler.future.CompletableFutureWithCounter;
import ru.yandex.juggler.relay.JugglerHttpRelayFactory;
import ru.yandex.juggler.relay.JugglerRelay;
import ru.yandex.juggler.relay.JugglerRelayFactory;
import ru.yandex.juggler.target.JugglerTarget;
import ru.yandex.juggler.target.JugglerTargetFactory;
import ru.yandex.juggler.target.JugglerTargetOptions;
import ru.yandex.juggler.target.TargetState;
import ru.yandex.juggler.validation.JugglerValidator;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class JugglerDirectClient implements JugglerClient {
    private final FeatureFlagsHolder flagsHolder;
    private static final Logger logger = LoggerFactory.getLogger(JugglerDirectClient.class);

    private volatile Map<String, JugglerRelay> relaysByAddress = Map.of();
    private volatile Map<String, JugglerTarget> targetsByName = Map.of();

    private final JugglerClientMetrics clientMetrics;
    private final JugglerRelayFactory relayFactory;
    private final JugglerTargetFactory targetFactory;
    private volatile boolean closed = false;

    public JugglerDirectClient(JugglerDirectClientOptions clientOptions, JugglerTargetOptions targetOptions, FeatureFlagsHolder flagsHolder) {
        MetricRegistry registry = Objects.requireNonNull(clientOptions.registry, "registry");

        this.clientMetrics = new JugglerClientMetrics();

        var httpClientBuilder = HttpClient.newBuilder()
                .connectTimeout(clientOptions.relayConnectTimeout);

        if (clientOptions.relayHttpExecutor != null) {
            httpClientBuilder.executor(clientOptions.relayHttpExecutor);
        }

        HttpClient relayHttpClient = httpClientBuilder.build();
        this.relayFactory = new JugglerHttpRelayFactory(relayHttpClient, registry,
                clientOptions.relayResponseTimeout,
                clientOptions.relayCircuitBreakerFactory);

        this.targetFactory = (name, initialState, isRegional) -> new JugglerTarget(name, initialState, clientMetrics, targetOptions, isRegional);
        this.flagsHolder = flagsHolder;
    }

    @Override
    synchronized public void updateTargetConfig(List<GetConfigResponse.Target> targetsConfig) {
        if (closed) {
            return;
        }

        final HashMap<String, JugglerRelay> currentRelays = new HashMap<>(this.relaysByAddress);
        final HashMap<String, JugglerTarget> currentTargets = new HashMap<>(this.targetsByName);

        // 1. Upsert new targets
        Set<String> newTargetNames = new HashSet<>(targetsConfig.size());
        for (var targetConfig : targetsConfig) {
            String targetName = targetConfig.name;
            List<GetConfigResponse.Relay> relays = targetConfig.relays;
            boolean isRegional = targetConfig.isRegional;

            newTargetNames.add(targetName);

            ArrayList<JugglerRelay> targetRelays = new ArrayList<>(relays.size());
            IntArrayList weights = new IntArrayList(relays.size());
            for (var relay : relays) {
                weights.add(relay.weight);
                targetRelays.add(getOrCreateRelay(targetName, currentRelays, relay.address));
            }

            JugglerTarget target = currentTargets.get(targetName);
            TargetState state = new TargetState(targetRelays, weights);
            if (target != null) {
                target.update(state);
            } else {
                currentTargets.put(targetName, targetFactory.makeTarget(targetName, state, isRegional));
            }
        }

        this.relaysByAddress = Map.copyOf(currentRelays);

        // 2. Close and filter out missing targets
        this.targetsByName = currentTargets.entrySet().stream()
                .filter(entry -> {
                    if (!newTargetNames.contains(entry.getKey())) {
                        entry.getValue().close();
                        return false;
                    }
                    return true;
                })
                .collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private JugglerRelay getOrCreateRelay(String targetName, Map<String, JugglerRelay> relays, String address) {
        var relay = relays.get(address);
        if (relay != null) {
            return relay;
        }

        return relays.computeIfAbsent(address, ignore -> relayFactory.makeRelay(targetName, address));
    }

    @Override
    public CompletableFuture<EventStatus> sendEvent(JugglerEvent event) {
        var validateStatus = JugglerValidator.validateEvent(event);
        if (validateStatus != null) {
            return CompletableFuture.completedFuture(validateStatus);
        }

        var targets = targetsByName;
        if (targets.size() == 0) {
            return CompletableFuture.completedFuture(new EventStatus(
                    "No juggler targets are resolved yet", HttpStatus.SC_503_SERVICE_UNAVAILABLE
            ));
        }

        if (targets.size() == 1) {
            return Iterables.getOnlyElement(targets.values())
                    .addEvent(event)
                    .exceptionally(EventStatus::new);
        }

        String projectId = event.projectId;
        boolean globalDisabled = flagsHolder.hasFlag(FeatureFlag.DISABLE_GLOBAL_JUGGLER, projectId);
        var regionalTargets = onlyRegional(targets.values());
        List<CompletableFuture<EventStatus>> futures = (globalDisabled ? regionalTargets : targets.values()).stream()
                .map(target -> target.addEvent(event))
                .map(future -> future.exceptionally(EventStatus::new))
                .collect(Collectors.toList());
        if (futures.isEmpty()) {
            return CompletableFuture.completedFuture(new EventStatus(
                    "No juggler targets are resolved yet", HttpStatus.SC_503_SERVICE_UNAVAILABLE
            ));
        }
        if (globalDisabled || !hasGlobal(targets.values())) {
            var union = new CompletableFutureWithCounter<>(futures);
            return union.thenApply(this::mergeStatuses);
        }
        return CompletableFutures.allOf(futures).thenApply(this::mergeStatuses);
    }

    private static boolean hasGlobal(Collection<JugglerTarget> targets) {
        for (var target : targets) {
            if (!target.isRegional()) {
                return true;
            }
        }
        return false;
    }

    private static List<JugglerTarget> onlyRegional(Collection<JugglerTarget> targets) {
        List<JugglerTarget> regionalTargets = new ArrayList<>();
        for (var target : targets) {
            if (target.isRegional()) {
                regionalTargets.add(target);
            }
        }
        return regionalTargets;
    }

    private EventStatus mergeStatuses(List<EventStatus> eventStatuses) {
        var anySuccess = eventStatuses.stream()
                .filter(EventStatus::isOk)
                .findFirst();
        if (anySuccess.isPresent()) {
            return anySuccess.get();
        }

        if (eventStatuses.isEmpty()) {
            logger.error("Cross target send completed with no events");
            return new EventStatus("No juggler target succeeded", 500);
        }

        return eventStatuses.get(0);
    }

    @Override
    public void close() {
        closed = true;
        for (var target : targetsByName.values()) {
            target.close();
        }
    }

    @Override
    public int estimateCount() {
        return clientMetrics.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        clientMetrics.append(tsMillis, commonLabels, consumer);
    }
}
