package ru.yandex.juggler.future;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class CompletableFutureWithCounter<T extends StatusAware> extends CompletableFuture<List<T>> {
    private final List<CompletableFuture<T>> futures;
    private final AtomicInteger okToComplete;
    private final AtomicInteger allToComplete;

    public CompletableFutureWithCounter(List<CompletableFuture<T>> futures) {
        this.futures = futures;
        int n = futures.size();
        if (n == 0) {
            throw new IllegalArgumentException("Empty list of futures supplied to CompletableFutureWithCounter");
        }
        this.allToComplete = new AtomicInteger(n);
        this.okToComplete = new AtomicInteger((n + 1) / 2);

        for (var future : futures) {
            future.whenComplete(this::oneCompleted);
        }
    }

    private void oneCompleted(@Nullable T result, @Nullable Throwable throwable) {
        if (allToComplete.decrementAndGet() == 0) {
            completeWithReady();
            return;
        }

        boolean completedOk = (throwable == null) && (result != null) && result.isOk();
        if (completedOk && okToComplete.decrementAndGet() == 0) {
            completeWithReady();
        }
    }

    private void completeWithReady() {
        if (isDone()) {
            return;
        }

        complete(getReady());
    }

    public List<T> getReady() {
        List<T> completed = new ArrayList<>(futures.size());
        for (var future : futures) {
            if (future.isDone() && !future.isCompletedExceptionally()) {
                completed.add(future.getNow(null));
            }
        }
        return completed;
    }
}
