package ru.yandex.juggler.relay;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.juggler.dto.EventStatus;
import ru.yandex.juggler.dto.JugglerBatchRequest;
import ru.yandex.juggler.dto.JugglerBatchResponse;
import ru.yandex.juggler.dto.JugglerEvent;
import ru.yandex.juggler.exceptions.BadGatewayException;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.failsafe.CircuitBreaker;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class JugglerHttpRelay implements JugglerRelay {
    private static final Logger logger = LoggerFactory.getLogger(JugglerHttpRelay.class);

    private final URI endpointUrl;
    private final HttpClient httpClient;
    private final HttpRelayMetrics metrics;
    private final ObjectMapper mapper = new ObjectMapper();

    private final String address;
    private final CircuitBreaker circuitBreaker;
    private final Duration timeout;

    public JugglerHttpRelay(String address, HttpClient httpClient, MetricRegistry registry, Duration timeout, CircuitBreaker circuitBreaker) {
        this.address = address;
        this.endpointUrl = URI.create("http://" + address + "/events");
        this.httpClient = httpClient;
        this.metrics = new HttpRelayMetrics(registry);
        this.timeout = timeout;
        this.circuitBreaker = circuitBreaker;
    }

    @Override
    public boolean attemptServe() {
        if (!circuitBreaker.attemptExecution()) {
            logger.debug("CircuitBreaker is open for " + address);
            metrics.circuitBreakerDisallows.inc();
            return false;
        }
        return true;
    }

    @Override
    // sendBatch must only be called after successful attemptServe
    public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName, List<JugglerEvent> events) {
        return CompletableFutures.safeCall(() -> sendBatchUnsafe(sourceAppName, events))
                .whenComplete(this::updateStatusAndCircuitBreaker)
                .thenApply(httpResponse -> extractEventStatuses(httpResponse, events.size()));
    }

    public CompletableFuture<HttpResponse<byte[]>> sendBatchUnsafe(String sourceAppName, List<JugglerEvent> events) throws Exception {
        var batchRequest = new JugglerBatchRequest(sourceAppName, events);
        HttpRequest request = HttpRequest.newBuilder(endpointUrl)
                .POST(HttpRequest.BodyPublishers.ofString(mapper.writeValueAsString(batchRequest)))
                .timeout(timeout)
                .build();
        return metrics.eventsMetrics.callMetrics.wrapFuture(httpClient
                .sendAsync(request, HttpResponse.BodyHandlers.ofByteArray())
                .orTimeout(timeout.toMillis(), TimeUnit.MILLISECONDS)); // if HttpClient gets stuck for whatever reason
    }

    private List<EventStatus> extractEventStatuses(HttpResponse<byte[]> httpResponse, int expectedEvents) {
        JugglerBatchResponse response;
        try {
            response = mapper.readValue(httpResponse.body(), JugglerBatchResponse.class);
        } catch (IOException e) {
            logger.error("Failed to deserialize response in " + endpointUrl + ": " + new String(httpResponse.body()), e);
            throw new BadGatewayException("Failed to deserialize response in " + endpointUrl, e);
        }

        if (response.success != null && !response.success) {
            String details = Nullables.orDefault(response.message,
                    "Call to " + endpointUrl + " was unsuccessful without details");
            logger.error(details);
            throw new BadGatewayException(details);
        }
        var eventStatuses = Nullables.orEmpty(response.events);
        if (eventStatuses.size() != expectedEvents) {
            String details = "Call to " + endpointUrl + " returned incorrect number of events: " +
                    "expected " + expectedEvents + ", got " + eventStatuses.size();
            logger.error(details);
            throw new BadGatewayException(details);
        }
        return eventStatuses;
    }

    private void updateStatusAndCircuitBreaker(@Nullable HttpResponse<byte[]> response, @Nullable Throwable exception) {
        // TODO: Copy-paste until SOLOMON-5279
        if (exception != null) {
            circuitBreaker.markFailure();
            logger.error("Call to " + endpointUrl + " failed", exception);
            return;
        }

        if (response != null) {
            int status = response.statusCode();
            metrics.eventsMetrics.incStatus(status);
            if (HttpStatus.is5xx(status)) {
                circuitBreaker.markFailure();
            } else {
                circuitBreaker.markSuccess();
            }
        }
    }
}
