package ru.yandex.juggler.resolver;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.juggler.dto.GetConfigRequest;
import ru.yandex.juggler.dto.GetConfigResponse;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.failsafe.CircuitBreaker;
import ru.yandex.solomon.selfmon.failsafe.ExpMovingAverageCircuitBreaker;
import ru.yandex.solomon.selfmon.http.HttpClientMetrics;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.file.FileStorage;
import ru.yandex.solomon.util.host.HostUtils;

import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class HttpProxyResolver implements AutoCloseable, ProxyResolver {
    private final Logger logger = LoggerFactory.getLogger(HttpProxyResolver.class);

    @VisibleForTesting
    public static final String STATE_FILE = "juggler.http_proxy_resolver.state";

    private final FileStorage storage;
    private final String global;
    private final String regional;
    private final HttpClient httpClient;
    private final HttpClientMetrics metrics;
    private final ObjectMapper mapper = new ObjectMapper();
    private final CircuitBreaker circuitBreaker =
            new ExpMovingAverageCircuitBreaker(0.4, TimeUnit.MILLISECONDS.toMillis(30));

    private final Duration requestTimeout;
    private final String hostname;

    private final ProxyResolveObserver observer;
    private final PingActorRunner resolveRelaysActor;

    @VisibleForTesting
    public HttpProxyResolver(
            FileStorage storage,
            String global,
            String regional,
            MetricRegistry registry,
            ScheduledExecutorService timer,
            Executor executor,
            ProxyResolveObserver resolveObserver)
    {
        this(storage, global, regional, registry, timer, executor, resolveObserver,
                Duration.ofSeconds(60), Duration.ofSeconds(15), Duration.ofSeconds(15),
                HostUtils.getFqdn());
    }

    public HttpProxyResolver(
            FileStorage storage,
            String global,
            String regional,
            MetricRegistry registry,
            ScheduledExecutorService timer,
            Executor executor,
            ProxyResolveObserver observer,
            Duration refreshInterval,
            Duration connectTimeout,
            Duration requestTimeout,
            String hostname)
    {
        this.storage = storage;
        this.hostname = hostname;
        this.httpClient = HttpClient.newBuilder()
                .connectTimeout(connectTimeout)
                .executor(executor)
                .followRedirects(HttpClient.Redirect.NEVER)
                .version(HttpClient.Version.HTTP_1_1)
                .build();
        this.global = global;
        this.regional = regional;

        this.metrics = new HttpClientMetrics("juggler-proxy-resolver", registry);
        this.requestTimeout = requestTimeout;

        this.resolveRelaysActor = PingActorRunner.newBuilder()
            .onPing(this::act)
            .operation("resolve_relays")
            .timer(timer)
            .executor(executor)
            .pingInterval(refreshInterval)
            .backoffDelay(Duration.ofSeconds(10))
            .backoffMaxDelay(Duration.ofMinutes(5))
            .build();

        this.observer = observer;

        loadConfigFromFile();
        resolveRelaysActor.schedule();
    }

    private CompletableFuture<Void> act(int attempt) {
        return safeCall(this::resolveRelays);
    }

    private CompletableFuture<Void> resolveRelays() {
        var regionalConfig = fetchConfig(regional)
                .exceptionally(e -> {
                    logger.error("regional fetch configuration exception", e);
                    return List.of();
                });
        CompletableFuture<List<GetConfigResponse.Target>> result;
        if (global.isBlank()) {
            result = regionalConfig.thenApply(this::markRegional);
        } else {
            var globalConfig = fetchConfig(global)
                    .exceptionally(e -> {
                        logger.error("global fetch configuration exception", e);
                        return List.of();
                    });
            var configsFuture = CompletableFutures.allOf2(globalConfig, regionalConfig);
            result = configsFuture.thenApply(
                configs -> configs.reduce(
                    (globalList, regionalList) -> {
                        var list = new ArrayList<GetConfigResponse.Target>(globalList.size() + regionalList.size());
                        list.addAll(markRegional(regionalList));
                        list.addAll(globalList); // global goes last to win in case of a collision
                        return list;
                    }));
        }
        return result.thenAccept(config -> {
            if (config != null && !config.isEmpty()) {
                var observer = this.observer;
                if (observer != null) {
                    observer.updateTargetConfig(config);
                }

                saveConfigToFile(config);
            }
        })
                .exceptionally(e -> {
                    logger.error("Processing juggler targets config update failed", e);
                    return null;
                });
    }

    private List<GetConfigResponse.Target> markRegional(List<GetConfigResponse.Target> targets) {
        List<GetConfigResponse.Target> list = new ArrayList<>(targets.size());
        for (var target : targets) {
            var newTarget = new GetConfigResponse.Target(target.name, target.relays, true);
            list.add(newTarget);
        }
        return list;
    }

    private CompletableFuture<List<GetConfigResponse.Target>> fetchConfig(String basePath) {
        if (!circuitBreaker.attemptExecution()) {
            return failedFuture(new RuntimeException("CircuitBreaker is open for " + basePath));
        }

        var endpoint = "/v2/agent/get_config";
        var url = URI.create(basePath + endpoint);

        GetConfigRequest body = new GetConfigRequest(hostname);
        try {
            var request = HttpRequest.newBuilder(url)
                    .POST(HttpRequest.BodyPublishers.ofString(mapper.writeValueAsString(body)));
            return call(basePath, endpoint, request, GetConfigResponse.class)
                    .thenApply(rs -> Nullables.orEmpty(rs.targets));
        } catch (JsonProcessingException e) {
            return failedFuture(e);
        }
    }

    private <T> CompletableFuture<T> call(
            String basePath,
            String endpointName,
            HttpRequest.Builder requestBuilder,
            Class<T> clazz)
    {
        var request = requestBuilder
                .header("X-Request-Id", UUID.randomUUID().toString())
                .header("Content-Type", "application/json")
                .header("Accept", "application/json")
                .timeout(requestTimeout)
                .build();
        var endpoint = metrics.endpoint(endpointName);
        String url = basePath + endpointName;
        try {
            CompletableFuture<HttpResponse<byte[]>> apiCall = endpoint.callMetrics.wrapFuture(() ->
                    httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray()));

            return apiCall.whenComplete((response, exception) -> {
                if (exception != null) {
                    circuitBreaker.markFailure();
                    logger.error("Call to " + url + " failed", exception);
                }

                if (response != null) {
                    int status = response.statusCode();
                    endpoint.incStatus(status);
                    if (HttpStatus.is5xx(status)) {
                        circuitBreaker.markFailure();
                    } else {
                        circuitBreaker.markSuccess();
                    }
                }
            })
            .thenApply(httpResponse -> {
                if (!HttpStatus.is2xx(httpResponse.statusCode())) {
                    throw new IllegalStateException("Bad return status " + httpResponse.statusCode() + " for " + url +
                            ": " + new String(httpResponse.body(), StandardCharsets.UTF_8)) {
                    };
                }
                try {
                    return mapper.readValue(httpResponse.body(), clazz);
                } catch (IOException e) {
                    throw new UncheckedIOException("Failed to deserialize response in " + url, e);
                }
            });
        } catch (Exception e) {
            circuitBreaker.markFailure();
            logger.error("Unhandled exception in " + endpointName, e);
            return failedFuture(e);
        }
    }

    private void saveConfigToFile(List<GetConfigResponse.Target> config) {
        try {
            storage.save(STATE_FILE, config, mapper::writeValueAsString);
        } catch (Throwable t) {
            logger.error("Error while writing state file " + STATE_FILE, t);
        }
    }

    private void loadConfigFromFile() {
        try {
            List<GetConfigResponse.Target> config = storage.load(
                STATE_FILE,
                state -> mapper.readValue(state, new TypeReference<List<GetConfigResponse.Target>>() {}));

            if (config == null) {
                return;
            }

            observer.updateTargetConfig(config);
        } catch (Throwable t) {
            logger.error("Error while reading state file " + STATE_FILE, t);
        }
    }

    @Override
    public void close() {
        resolveRelaysActor.close();
    }
}
