package ru.yandex.juggler.target;

import java.time.Clock;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.juggler.client.JugglerClientMetrics;
import ru.yandex.juggler.dto.EventStatus;
import ru.yandex.juggler.dto.JugglerEvent;
import ru.yandex.juggler.exceptions.JugglerRuntimeException;
import ru.yandex.juggler.exceptions.UnavailableException;
import ru.yandex.juggler.relay.JugglerRelay;
import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class JugglerTarget implements AutoCloseable {
    private final static Logger logger = LoggerFactory.getLogger(JugglerTarget.class);

    private volatile TargetState state;
    private volatile boolean closed = false;

    private final boolean isRegional;
    private final String name;
    private final ScheduledExecutorService timer;
    private final BlockingQueue<SendOperation> queue;
    private final TargetMetrics metrics;
    private final long flushIntervalMillis;
    private final String applicationName;
    private final int batchSizeHint;
    private final RetryConfig retryBatchConfig;
    private final Clock clock;

    private final ActorRunner actor;
    private final ScheduledFuture<?> scheduledFuture;
    private volatile long nextFlushTimeMillis = 0;

    public JugglerTarget(String name, TargetState initialState, JugglerClientMetrics metrics, JugglerTargetOptions options) {
        this(name, initialState, metrics, options, false);
    }

    public JugglerTarget(String name, TargetState initialState, JugglerClientMetrics metrics, JugglerTargetOptions options, boolean isRegional) {
        this.name = name;
        this.state = initialState;
        this.timer = Objects.requireNonNull(options.timer, "timer");
        this.queue = new ArrayBlockingQueue<>(options.queueCapacity);
        this.batchSizeHint = options.batchSizeHint;
        this.metrics = metrics.getMetricsForTarget(name);
        this.flushIntervalMillis = options.flushIntervalMillis;
        this.applicationName = options.applicationName;
        this.retryBatchConfig = RetryConfig.DEFAULT
                .withNumRetries(options.maxSendRetries)
                .withDelay(options.retryDelayMillis);
        this.clock = Clock.systemUTC();
        this.isRegional = isRegional;

        var flusherExecutor = Objects.requireNonNull(options.flusherExecutor, "flusherExecutor");
        this.actor = new ActorRunner(this::flush, flusherExecutor);
        long scheduleDelay = flushIntervalMillis / 2;
        this.scheduledFuture = timer.scheduleWithFixedDelay(actor::schedule, scheduleDelay, scheduleDelay, TimeUnit.MILLISECONDS);
    }

    private long getFlushDelay() {
        return flushIntervalMillis / 2 + ThreadLocalRandom.current().nextLong(flushIntervalMillis);
    }

    private void flush() {
        if (closed) {
            List<SendOperation> batch = new ArrayList<>(queue.size());
            queue.drainTo(batch);
            metrics.updateQueueSize(queue.size());
            if (!batch.isEmpty()) {
                EventStatus targetClosed = error("Event discarded, target " + name + " is closed", HttpStatus.SC_404_NOT_FOUND);
                batch.forEach(op -> op.complete(targetClosed));
            }
            scheduledFuture.cancel(false);
            return;
        }

        long now = clock.millis();
        boolean needFlush = now > nextFlushTimeMillis || queue.size() > batchSizeHint;
        if (!needFlush) {
            return;
        }

        List<SendOperation> batch = new ArrayList<>(2 * batchSizeHint);
        queue.drainTo(batch, 2 * batchSizeHint);
        metrics.updateQueueSize(queue.size());
        if (!batch.isEmpty()) {
            sendBatch(batch);
        }
        nextFlushTimeMillis = now + getFlushDelay();
    }

    public CompletableFuture<EventStatus> addEvent(JugglerEvent event) {
        if (closed) {
            return CompletableFuture.completedFuture(error(
                    "Target " + name + " is closed",
                    HttpStatus.SC_404_NOT_FOUND
            ));
        }

        CompletableFuture<EventStatus> promise = new CompletableFuture<>();
        if (!queue.offer(new SendOperation(event, promise))) {
            metrics.overflow();
            promise.complete(error(
                    "Target " + name + " event queue overflow",
                    HttpStatus.SC_429_TOO_MANY_REQUESTS
            ));
            return promise;
        }

        metrics.updateQueueSize(queue.size());
        actor.schedule();

        return promise;
    }

    private void sendBatch(List<SendOperation> batch) {
        metrics.batchStarted();
        metrics.eventsStarted(batch.size());
        RetryCompletableFuture.runWithRetries(() -> doSendBatch(batch), retryBatchConfig)
                .whenComplete((response, error) -> completeBatch(batch, response, error));
    }

    private CompletableFuture<List<EventStatus>> doSendBatch(List<SendOperation> batch) {
        List<JugglerEvent> events = batch.stream()
                .map(SendOperation::getEvent)
                .collect(Collectors.toList());

        Optional<JugglerRelay> maybeRelay = state.chooseRelay();

        if (maybeRelay.isEmpty()) {
            String message = "Failed to choose a working relay for target " + name;
            logger.error(message);
            return CompletableFuture.failedFuture(new UnavailableException(message));
        }

        return CompletableFutures.safeCall(() -> maybeRelay.get()
                .sendBatch(applicationName, events))
                .exceptionally(throwable -> {
                    logger.error("Unhandled exception in doSendBatch", throwable);
                    throw new JugglerRuntimeException(throwable.getMessage(), throwable, HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
                });
    }

    private void completeBatch(
            List<SendOperation> batch,
            List<EventStatus> response,
            @Nullable Throwable error)
    {
        if (error != null) {
            Throwable cause = CompletableFutures.unwrapCompletionException(error);
            EventStatus status;
            if (cause instanceof JugglerRuntimeException) {
                status = ((JugglerRuntimeException) cause).toEventStatus();
            } else {
                String message = cause.getClass().getSimpleName() + ": " + cause.getMessage();
                status = error(message, HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
            }
            for (var operation : batch) {
                operation.complete(status);
            }
            metrics.batchFailed();
            metrics.eventsFailed(batch.size());
            return;
        }

        int eventsOk = 0, eventsFailed = 0;
        for (int i = 0; i < batch.size(); i++) {
            var status = response.get(i);
            batch.get(i).complete(status);
            if (status.code == HttpStatus.SC_200_OK) {
                eventsOk++;
            } else {
                eventsFailed++;
            }
        }
        metrics.eventsOk(eventsOk);
        metrics.eventsFailed(eventsFailed);
        metrics.batchSucceeded();
    }

    public void update(TargetState newState) {
        state = newState;
    }

    private EventStatus error(String message, int code) {
        var status = new EventStatus();

        status.code = code;
        status.message = message;

        return status;
    }

    @Override
    public void close() {
        closed = true;
        actor.schedule();
    }

    public boolean isRegional() {
        return isRegional;
    }
}
