package ru.yandex.juggler.target;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.primitives.LazyGaugeDouble;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class TargetMetrics implements MetricSupplier {
    private final MetricRegistry registry;

    private final Rate queueOverflows;
    private final GaugeDouble queueSize;

    private final Rate eventsStarted;
    private final Rate eventsSuccess;
    private final Rate eventsFailed;
    private final LazyGaugeDouble eventsInFlight;

    private final Rate requestsStarted;
    private final Rate requestsSuccess;
    private final Rate requestsFailed;
    private final LazyGaugeDouble requestsInFlight;

    public TargetMetrics(String targetName) {
        registry = targetName.isEmpty()
                ? new MetricRegistry()
                : new MetricRegistry(Labels.of("jugglerTarget", targetName));

        queueOverflows = registry.rate("jugglerClient.sendOverflow");

        queueSize = registry.gaugeDouble("jugglerClient.sendQueueSize");

        eventsStarted = registry.rate("jugglerClient.events.started");
        eventsSuccess = registry.rate("jugglerClient.events.success");
        eventsFailed = registry.rate("jugglerClient.events.failed");
        eventsInFlight = registry.lazyGaugeDouble("jugglerClient.events.inFlight",
                () -> eventsStarted.get() - eventsSuccess.get() - eventsFailed.get());

        requestsStarted = registry.rate("jugglerClient.requests.started");
        requestsSuccess = registry.rate("jugglerClient.requests.success");
        requestsFailed = registry.rate("jugglerClient.requests.failed");
        requestsInFlight = this.registry.lazyGaugeDouble("jugglerClient.requests.inFlight",
                () -> requestsStarted.get() - requestsSuccess.get() - requestsFailed.get());
    }

    public void updateQueueSize(int queueSize) {
        this.queueSize.set(queueSize);
    }

    public void overflow() {
        this.queueOverflows.inc();
    }

    public void batchStarted() {
        this.requestsStarted.inc();
    }

    public void batchFailed() {
        this.requestsFailed.inc();
    }

    public void batchSucceeded() {
        this.requestsSuccess.inc();
    }

    public void eventsStarted(int eventsStarted) {
        if (eventsStarted > 0) {
            this.eventsStarted.add(eventsStarted);
        }
    }

    public void eventsOk(int eventsOk) {
        if (eventsOk > 0) {
            this.eventsSuccess.add(eventsOk);
        }
    }

    public void eventsFailed(int eventsFailed) {
        if (eventsFailed > 0) {
            this.eventsFailed.add(eventsFailed);
        }
    }

    public void combine(TargetMetrics other) {
        queueOverflows.combine(other.queueOverflows);
        queueSize.combine(other.queueSize);

        eventsStarted.combine(other.eventsStarted);
        eventsSuccess.combine(other.eventsSuccess);
        eventsFailed.combine(other.eventsFailed);

        requestsStarted.combine(other.requestsStarted);
        requestsSuccess.combine(other.requestsSuccess);
        requestsFailed.combine(other.requestsFailed);
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }
}
