package ru.yandex.juggler.target;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.IntArrayList;

import ru.yandex.juggler.relay.CircuitBreakingAware;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class WeightedChoice<T extends CircuitBreakingAware> {
    private final List<T> backends;
    private final int[] cummulativeWeights;
    private final Random random;
    private final int maxRetriesForReadyCheck;

    public WeightedChoice(List<T> backends, List<Integer> weights, Random random, int maxRetriesForReadyCheck) {
        this.backends = new ArrayList<>(backends.size());
        IntArrayList cummulativeWeightList = new IntArrayList(backends.size());
        int totalWeights = 0;
        for (int idx = 0; idx < backends.size(); idx++) {
            if (weights.get(idx) == 0) {
                continue;
            }
            this.backends.add(backends.get(idx));
            totalWeights += weights.get(idx);
            cummulativeWeightList.add(totalWeights);
        }
        this.cummulativeWeights = cummulativeWeightList.toIntArray();
        this.random = random;
        this.maxRetriesForReadyCheck = maxRetriesForReadyCheck;
    }

    public Optional<T> choose() {
        int n = backends.size();
        if (n == 0) {
            return Optional.empty();
        }
        int totalWeights = cummulativeWeights[n - 1];
        for (int attempt = 0; attempt < maxRetriesForReadyCheck; attempt++) {
            int loc = random.nextInt(totalWeights);
            int pos = upperBound(cummulativeWeights, loc);
            // Call attemptServe() only on that backend that we are ready to return
            T backend = backends.get(pos);
            if (backend.attemptServe()) {
                return Optional.of(backend);
            }
            // No luck, try to choose another backend, or maybe even this again.
            // CircuitBreaker may let us to make a single request even if we've
            // just got attemptServe() = false from it.
        }
        // all attempted random backends are in CircuitBreaker OPEN state and don't allow us to make a request
        return Optional.empty();
    }

    private int upperBound(int[] list, int key) {
        int count = list.length;
        int first = 0;

        while (count > 0) {
            int step = count / 2;
            int it = first + step;
            if (list[it] <= key) {
                first = ++it;
                count -= step + 1;
            } else {
                count = step;
            }
        }
        return first;
    }
}
