package ru.yandex.juggler.validation;

import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.juggler.dto.EventStatus;
import ru.yandex.juggler.dto.JugglerEvent;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * https://st.yandex-team.ru/JUGGLER-2270#1490700974000
 * https://a.yandex-team.ru/arc/trunk/arcadia/juggler/libjuggler/validators/system.py?rev=r8572230
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class JugglerValidator {
    private static final int MAX_HOST_NAME_LENGTH = 192;
    private static final int MAX_SERVICE_NAME_LENGTH = 128;
    private static final int MAX_TAG_NAME_LENGTH = 128;
    private static final int MAX_INSTANCE_NAME_LEN = 126;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;

    private static final Pattern VALID_HOST_NAME_PATTERN = Pattern.compile("^(([a-zA-Z0-9]|[-a-zA-Z0-9][\\w\\-]*[a-zA-Z0-9])\\.)*([a-zA-Z0-9]|[-a-zA-Z0-9][\\w\\-]*[a-zA-Z0-9])$");
    private static final Pattern VALID_SERVICE_NAME_PATTERN = Pattern.compile("^[\\w\\-+\\./]+$");
    private static final Pattern VALID_TAG_NAME_PATTERN = Pattern.compile("^[\\w\\-+\\._]+$");
    private static final Pattern VALID_INSTANCE_NAME_PATTERN = VALID_SERVICE_NAME_PATTERN;

    private JugglerValidator() {
    }

    public static boolean isHostNameValid(String hostName) {
        return hostName.length() <= MAX_HOST_NAME_LENGTH && isValid(VALID_HOST_NAME_PATTERN, hostName);
    }

    public static boolean isServiceNameValid(String serviceName) {
        return serviceName.length() <= MAX_SERVICE_NAME_LENGTH && isValid(VALID_SERVICE_NAME_PATTERN, serviceName);
    }

    public static boolean isValidTagName(String tag) {
        return tag.length() <= MAX_TAG_NAME_LENGTH && isValid(VALID_TAG_NAME_PATTERN, tag);
    }

    public static boolean isValidDescription(String description) {
        return description.length() <= MAX_DESCRIPTION_LENGTH;
    }

    private static boolean isValid(Pattern pattern, String str) {
        Matcher matcher = pattern.matcher(str);
        return matcher.matches();
    }

    public static boolean isInstanceNameValid(String instance) {
        if (instance.isEmpty()) {
            return true;
        }

        return instance.length() < MAX_INSTANCE_NAME_LEN && isValid(VALID_INSTANCE_NAME_PATTERN, instance);
    }

    public static EventStatus validateEvent(JugglerEvent event) {
        EventStatus status;

        if ((status = JugglerEventField.HOST.validate(event.host)) != null) {
            return status;
        }

        if ((status = JugglerEventField.SERVICE.validate(event.service)) != null) {
            return status;
        }

        if (((status = JugglerEventField.INSTANCE.validate(event.instance))) != null) {
            return status;
        }

        for (String tag : event.tags) {
            if ((status = JugglerEventField.TAG.validate(tag)) != null) {
                return status;
            }
        }

        /*
        // Descripion validation is disabled since many channels violate the length <= 1024 condition. However, Juggler seems to accept that
        // TODO(uranix): add metric, enforce truncation
        if ((status = JugglerEventField.DESCRIPTION.validate(event.description)) != null) {
            return status;
        }
        */

        return null;
    }

    private static EventStatus error(String message, int code) {
        return new EventStatus(message, code);
    }

    private enum JugglerEventField {
        HOST(true, JugglerValidator::isHostNameValid),
        SERVICE(true, JugglerValidator::isServiceNameValid),
        INSTANCE(false, JugglerValidator::isInstanceNameValid),
        TAG(true, JugglerValidator::isValidTagName),
        DESCRIPTION(false, JugglerValidator::isValidDescription);


        private final boolean required;
        private final Predicate<String> validator;
        private final Cache<String, Boolean> cache;
        private final Rate cacheHit;
        private final Rate cacheMiss;

        JugglerEventField(boolean required, Predicate<String> validator) {
            this.required = required;
            this.validator = validator;
            this.cache = CacheBuilder.newBuilder()
                    .expireAfterAccess(2, TimeUnit.MINUTES)
                    .maximumSize(1_000_000)
                    .build();
            this.cacheHit = cacheRate("Hit", this.name());
            this.cacheMiss = cacheRate("Miss", this.name());
        }

        public EventStatus validate(String value) {
            if (Strings.isNullOrEmpty(value)) {
                if (!required) {
                    return null;
                }

                return error(name() + "is empty", HttpStatus.SC_422_UNPROCESSABLE_ENTITY);
            }

            var valid = cache.getIfPresent(value);
            if (valid == null) {
                cacheMiss.inc();
                valid = validator.test(value);
                cache.put(value, valid);
            } else {
                cacheHit.inc();
            }

            if (!valid) {
                return error(name() + " is not valid: " + value, HttpStatus.SC_422_UNPROCESSABLE_ENTITY);
            }

            return null;
        }

        private static Rate cacheRate(String suffix, String field) {
            return MetricRegistry.root().rate("jugglerValidator.cache" + suffix, Labels.of("field", field));
        }
    }

}
