package ru.yandex.juggler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.juggler.client.JugglerDirectClient;
import ru.yandex.juggler.client.JugglerDirectClientOptions;
import ru.yandex.juggler.dto.EventStatus;
import ru.yandex.juggler.dto.GetConfigResponse;
import ru.yandex.juggler.dto.JugglerEvent;
import ru.yandex.juggler.resolver.HttpProxyResolver;
import ru.yandex.juggler.resolver.ProxyResolveObserver;
import ru.yandex.juggler.target.JugglerTargetOptions;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.util.file.DevNullFileStorage;

/**
 * @author Ivan Tsybulin
 */
public class Main {

    private static final String PROJECT_ID = "prudent_project_id";

    @ParametersAreNonnullByDefault
    private static class ObserverWrapper implements ProxyResolveObserver {
        private final ProxyResolveObserver observer;

        private ObserverWrapper(ProxyResolveObserver observer) {
            this.observer = observer;
        }

        @Override
        public void updateTargetConfig(List<GetConfigResponse.Target> config) {
            observer.updateTargetConfig(config);
            System.out.println("Got config: " + config);
        }
    }

    public static void main(String[] args) throws InterruptedException {
        MetricRegistry registry = new MetricRegistry();
        ScheduledExecutorService timer = Executors.newSingleThreadScheduledExecutor();
        ExecutorService executor = Executors.newFixedThreadPool(4);

        FeatureFlagHolderStub stub = new FeatureFlagHolderStub();
        stub.setFlag(PROJECT_ID, FeatureFlag.DISABLE_GLOBAL_JUGGLER, true);

        JugglerDirectClient directClient = new JugglerDirectClient(
                new JugglerDirectClientOptions()
                    .setRelayHttpExecutor(executor)
                    .setRegistry(registry),
                new JugglerTargetOptions()
                    .setFlusherExecutor(executor)
                    .setTimer(timer),
                stub
        );

        HttpProxyResolver resolver = new HttpProxyResolver(
                new DevNullFileStorage(),
                "http://juggler-api.search.yandex.net",
                "http://juggler-testing-api.search.yandex.net",
                registry,
                timer,
                executor,
                new ObserverWrapper(directClient));
        //ProxyResolver resolver = new SinglePushWrapperResolver("juggler-push.search.yandex.net:80");

        List<CompletableFuture<EventStatus>> futures = new ArrayList<>();

        Thread.sleep(3000);

        for (int i = 0; i < 10000; i++) {
            var event = directClient.sendEvent(new JugglerEvent("test-check-" + i, "test-service", "", JugglerEvent.Status.CRIT, "test event", List.of(), 0, PROJECT_ID));
            futures.add(event);
            if (ThreadLocalRandom.current().nextInt(30) < 1) {
                //Thread.sleep(50);
            }
        }

        var statuses = CompletableFutures.allOf(futures).join();

        Map<Integer, MutableInt> countByStatus = new HashMap<>();
        statuses.forEach(s -> countByStatus.computeIfAbsent(s.code, ignore -> new MutableInt()).increment());

        System.out.println(countByStatus);

        resolver.close();
        directClient.close();

        timer.shutdown();
        executor.shutdown();
        timer.awaitTermination(2, TimeUnit.SECONDS);
        executor.awaitTermination(2, TimeUnit.SECONDS);

        System.out.println("done");
    }
}
