package ru.yandex.juggler.http;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.Resources;
import org.apache.commons.lang3.mutable.Mutable;
import org.apache.commons.lang3.mutable.MutableObject;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;
import org.mockserver.junit.MockServerRule;
import org.mockserver.model.HttpRequest;
import org.mockserver.model.HttpResponse;

import ru.yandex.juggler.dto.GetConfigResponse;
import ru.yandex.juggler.resolver.HttpProxyResolver;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.file.SimpleFileStorage;

import static org.hamcrest.Matchers.greaterThan;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.mockserver.model.HttpStatusCode.NOT_FOUND_404;
import static ru.yandex.solomon.util.CloseableUtils.close;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class HttpProxyResolverTest {
    private HttpProxyResolver client;
    private MetricRegistry registry;
    private ScheduledExecutorService scheduler;
    private ExecutorService executor;

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    @Rule
    public TestName testName = new TestName();
    @Rule
    public final MockServerRule mockServerRule = new MockServerRule(this);

    private Mutable<List<GetConfigResponse.Target>> responseHolder;
    private CountDownLatch latch;

    private Path dir;

    @Before
    public void setUp() throws IOException {
        dir = tmp.newFolder(testName.getMethodName()).toPath();
        scheduler = Executors.newSingleThreadScheduledExecutor();
        executor = Executors.newFixedThreadPool(1);
        registry = new MetricRegistry();
        responseHolder = new MutableObject<>();
    }

    @After
    public void tearDown() {
        close(client);
        scheduler.shutdown();
        executor.shutdown();
    }

    @Test
    public void getConfig() throws InterruptedException {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(HttpResponse.response(load("get_config_response_regional.json")));
        var urlPrefix = "http://localhost:" + mockServerRule.getPort();

        latch = new CountDownLatch(1);
        client = new HttpProxyResolver(new SimpleFileStorage(dir), "", urlPrefix, registry, scheduler, executor,
                config -> {
                    responseHolder.setValue(config);
                    System.err.println(latch);
                    latch.countDown();
                },
                Duration.ofMillis(100), Duration.ofMillis(50), Duration.ofMillis(500),
                "solomon-dev-myt-00.search.yandex.net");

        latch.await();

        var response = responseHolder.getValue();
        assertEquals(3, response.size());
        for (var target : response) {
            assertEquals(14, target.relays.size());
            for (var relay : target.relays) {
                Assert.assertThat(relay.address.length(), greaterThan(0));
                Assert.assertThat(relay.weight, greaterThan(0));
            }
        }
    }

    @Test
    public void scheduling() throws InterruptedException {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(HttpResponse.response(load("get_config_response_regional.json")));
        var urlPrefix = "http://localhost:" + mockServerRule.getPort();

        latch = new CountDownLatch(3);
        client = new HttpProxyResolver(new SimpleFileStorage(dir), "", urlPrefix, registry, scheduler, executor,
                config -> {
                    responseHolder.setValue(config);
                    System.err.println(latch);
                    latch.countDown();
                },
                Duration.ofMillis(100), Duration.ofMillis(50), Duration.ofMillis(500),
                "solomon-dev-myt-00.search.yandex.net");

        latch.await();

        var response = responseHolder.getValue();
        assertEquals(3, response.size());
        for (var target : response) {
            assertEquals(14, target.relays.size());
            for (var relay : target.relays) {
                Assert.assertThat(relay.address.length(), greaterThan(0));
                Assert.assertThat(relay.weight, greaterThan(0));
            }
        }
    }

    @Test
    public void schedulingBothSuccess() throws InterruptedException {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/global/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(HttpResponse.response(load("get_config_response_global.json")));
        var urlPrefix1 = "http://localhost:" + mockServerRule.getPort() + "/global";
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/regional/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(HttpResponse.response(load("get_config_response_regional.json")));
        var urlPrefix2 = "http://localhost:" + mockServerRule.getPort() + "/regional";
        latch = new CountDownLatch(3);
        client = new HttpProxyResolver(new SimpleFileStorage(dir), urlPrefix1, urlPrefix2, registry, scheduler, executor,
                config -> {
                    responseHolder.setValue(config);
                    System.err.println(latch);
                    latch.countDown();
                },
                Duration.ofMillis(100), Duration.ofMillis(50), Duration.ofMillis(500),
                "solomon-dev-myt-00.search.yandex.net");

        latch.await();

        var response = responseHolder.getValue();
        assertEquals(6, response.size());
        int countRegional = 0;
        for (var target : response) {
            assertEquals(target.isRegional, target.name.startsWith("regional"));
            if (target.isRegional) {
                countRegional++;
            }
            assertEquals(14, target.relays.size());
            for (var relay : target.relays) {
                Assert.assertThat(relay.address.length(), greaterThan(0));
                Assert.assertThat(relay.weight, greaterThan(0));
            }
        }
        assertEquals(3, countRegional);
    }

    @Test
    public void schedulingBothSingleFail() throws InterruptedException {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/global/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(new HttpResponse().withStatusCode(NOT_FOUND_404.code()).withReasonPhrase(NOT_FOUND_404.reasonPhrase()));
        var urlPrefix1 = "http://localhost:" + mockServerRule.getPort() + "/global";
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/regional/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(HttpResponse.response(load("get_config_response_regional.json")));
        var urlPrefix2 = "http://localhost:" + mockServerRule.getPort() + "/regional";
        latch = new CountDownLatch(3);
        client = new HttpProxyResolver(new SimpleFileStorage(dir), urlPrefix1, urlPrefix2, registry, scheduler, executor,
                config -> {
                    responseHolder.setValue(config);
                    System.err.println(latch);
                    latch.countDown();
                },
                Duration.ofMillis(100), Duration.ofMillis(50), Duration.ofMillis(500),
                "solomon-dev-myt-00.search.yandex.net");

        latch.await();

        var response = responseHolder.getValue();
        assertEquals(3, response.size());
        int countRegional = 0;
        for (var target : response) {
            assertEquals(target.isRegional, target.name.startsWith("regional"));
            if (target.isRegional) {
                countRegional++;
            }
            assertEquals(14, target.relays.size());
            for (var relay : target.relays) {
                Assert.assertThat(relay.address.length(), greaterThan(0));
                Assert.assertThat(relay.weight, greaterThan(0));
            }
        }
        assertEquals(3, countRegional);
    }

    @Test
    public void schedulingBothFail() throws InterruptedException {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/global/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(new HttpResponse().withStatusCode(NOT_FOUND_404.code()).withReasonPhrase(NOT_FOUND_404.reasonPhrase()));
        var urlPrefix1 = "http://localhost:" + mockServerRule.getPort() + "/global";
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("POST")
                        .withPath("/regional/v2/agent/get_config")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json")
                        .withBody("{\"host_name\":\"solomon-dev-myt-00.search.yandex.net\"}"))
                .respond(new HttpResponse().withStatusCode(NOT_FOUND_404.code()).withReasonPhrase(NOT_FOUND_404.reasonPhrase()));
        var urlPrefix2 = "http://localhost:" + mockServerRule.getPort() + "/regional";
        latch = new CountDownLatch(3);
        client = new HttpProxyResolver(new SimpleFileStorage(dir), urlPrefix1, urlPrefix2, registry, scheduler, executor,
                config -> {
                    responseHolder.setValue(config);
                    System.err.println(latch);
                    latch.countDown();
                },
                Duration.ofMillis(100), Duration.ofMillis(50), Duration.ofMillis(500),
                "solomon-dev-myt-00.search.yandex.net");
        var response = responseHolder.getValue();
        assertNull(response);
        Assert.assertFalse(latch.await(1, TimeUnit.SECONDS));
    }

    @Test
    public void httpException() throws InterruptedException {
        var urlPrefix = "http://localhost:" + mockServerRule.getPort();
        latch = new CountDownLatch(1);
        client = new HttpProxyResolver(new SimpleFileStorage(dir), urlPrefix, "", registry, scheduler, executor,
                config -> {
                    responseHolder.setValue(config);
                    latch.countDown();
                },
                Duration.ofMillis(100), Duration.ofMillis(50), Duration.ofMillis(50),
                "solomon-dev-myt-00.search.yandex.net");

        Assert.assertFalse(latch.await(1, TimeUnit.SECONDS));
    }

    @Test
    public void loadFromDisk() throws Exception {
        // arrange
        var mapper = new ObjectMapper();
        var config = mapper.readValue(load("get_config_response_global.json"), GetConfigResponse.class).targets;

        var storage = new SimpleFileStorage(dir);
        storage.save(HttpProxyResolver.STATE_FILE, config, mapper::writeValueAsString);

        // act
        client = new HttpProxyResolver(
            storage,
            "http://localhost:" + mockServerRule.getPort(),
            "",
            registry,
            scheduler,
            executor,
            responseHolder::setValue);

        // assert
        var observed = responseHolder.getValue();
        assertEquals(3, observed.size());
        for (var target : observed) {
            assertEquals(14, target.relays.size());
            for (var relay : target.relays) {
                Assert.assertThat(relay.address.length(), greaterThan(0));
                Assert.assertThat(relay.weight, greaterThan(0));
            }
        }
    }

    private String load(String res) {
        try {
            URL url = HttpProxyResolverTest.class.getResource("json/" + res);
            return Resources.toString(url, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
