package ru.yandex.juggler.target;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.math3.stat.descriptive.SummaryStatistics;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.DisableOnDebug;
import org.junit.rules.TestRule;
import org.junit.rules.Timeout;

import ru.yandex.juggler.client.JugglerClientMetrics;
import ru.yandex.juggler.dto.EventStatus;
import ru.yandex.juggler.dto.JugglerEvent;
import ru.yandex.juggler.relay.JugglerRelay;
import ru.yandex.misc.concurrent.CompletableFutures;

import static org.hamcrest.CoreMatchers.containsString;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class JugglerTargetTest {
    @Rule
    public TestRule rule = new DisableOnDebug(Timeout.builder()
            .withTimeout(15, TimeUnit.SECONDS)
            .withLookingForStuckThread(true)
            .build());

    private JugglerClientMetrics metrics;
    private ScheduledExecutorService timer;
    private ExecutorService flusher;
    private JugglerTarget target;

    @Before
    public void setUp() {
        metrics = new JugglerClientMetrics();
        flusher = Executors.newFixedThreadPool(2);
        timer = Executors.newSingleThreadScheduledExecutor();
    }

    @After
    public void tearDown() {
        target.close();
        timer.shutdown();
        flusher.shutdown();
    }

    @Test
    public void empty() {
        target = new JugglerTarget("iva.juggler.search.yandex.net", TargetState.of(), metrics,
                new JugglerTargetOptions()
                    .setTimer(timer)
                    .setFlusherExecutor(flusher)
                    .setRetryDelayMillis(10)
                    .setFlushIntervalMillis(20));
        var promise = target.addEvent(new JugglerEvent(
                "solomon-alerting", "myservice", "", JugglerEvent.Status.CRIT, "The website is DOWN"));

        Assert.assertNotNull(promise);

        var response = promise.join();
        Assert.assertEquals(503, response.code);
    }

    @Test
    public void failingBackend() {
        JugglerRelay failing = new JugglerRelay() {
            @Override
            public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName,
                                                                  List<JugglerEvent> events) {
                return CompletableFuture.supplyAsync(() -> { throw new UnsupportedOperationException("I'm a teapot"); });
            }

            @Override
            public boolean attemptServe() {
                return true;
            }
        };

        target = new JugglerTarget("iva.juggler.search.yandex.net", TargetState.of(failing, 1), metrics,
                new JugglerTargetOptions()
                    .setTimer(timer)
                    .setFlusherExecutor(flusher)
                    .setRetryDelayMillis(10)
                    .setFlushIntervalMillis(20));
        var promise = target.addEvent(new JugglerEvent(
                "solomon-alerting", "myservice", "", JugglerEvent.Status.CRIT, "The website is DOWN"));

        Assert.assertNotNull(promise);

        var response = promise.join();
        Assert.assertEquals(500, response.code);
        Assert.assertThat(response.message, containsString("I'm a teapot"));
    }

    @Test
    public void randomFailing() throws Throwable {
        JugglerRelay failing = new JugglerRelay() {
            @Override
            public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName, List<JugglerEvent> events) {
                return CompletableFuture.supplyAsync(() -> {
                    System.out.println("attempt");
                    if (ThreadLocalRandom.current().nextInt(10) > 3) {
                        throw new UnsupportedOperationException("I'm a teapot");
                    } else {
                        return events.stream()
                                .map(event -> {
                                    var status = new EventStatus();
                                    status.code = 400;
                                    status.message = "Bad request";
                                    return status;
                                })
                                .collect(Collectors.toList());
                    }
                });
            }

            @Override
            public boolean attemptServe() {
                return true;
            }
        };

        target = new JugglerTarget("iva.juggler.search.yandex.net", TargetState.of(failing, 1), metrics,
                new JugglerTargetOptions()
                    .setTimer(timer)
                    .setFlusherExecutor(flusher)
                    .setMaxSendRetries(100)
                    .setRetryDelayMillis(0)
                    .setFlushIntervalMillis(20));
        var promise = target.addEvent(new JugglerEvent(
                "solomon-alerting", "myservice", "", JugglerEvent.Status.CRIT, "The website is DOWN"));

        Assert.assertNotNull(promise);

        try {
            promise.join();
        } catch (Throwable t) {
            throw CompletableFutures.unwrapCompletionException(t);
        }
    }

    @Test
    public void chooseGoodRelay() throws Throwable {
        JugglerRelay failing = new JugglerRelay() {
            @Override
            public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName,
                                                                  List<JugglerEvent> events) {
                System.out.println("nope");
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean attemptServe() {
                return true;
            }
        };

        JugglerRelay good = new JugglerRelay() {
            @Override
            public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName,
                                                                  List<JugglerEvent> events) {
                System.out.println("yep");
                return CompletableFuture.supplyAsync(() -> events.stream()
                        .map(ignore -> {
                            var status = new EventStatus();
                            status.code = 200;
                            return status;
                        })
                        .collect(Collectors.toList()));
            }

            @Override
            public boolean attemptServe() {
                return true;
            }
        };

        target = new JugglerTarget("iva.juggler.search.yandex.net", TargetState.of(failing, 4, good, 1),
                metrics,
                new JugglerTargetOptions()
                    .setTimer(timer)
                    .setFlusherExecutor(flusher)
                    .setMaxSendRetries(100)
                    .setRetryDelayMillis(0)
                    .setFlushIntervalMillis(20));
        var promise = target.addEvent(new JugglerEvent(
                "solomon-alerting", "myservice", "", JugglerEvent.Status.CRIT, "The website is DOWN"));

        Assert.assertNotNull(promise);

        try {
            promise.join();
        } catch (Throwable t) {
            throw CompletableFutures.unwrapCompletionException(t);
        }
    }

    @Test
    public void singleShotsAutoFlush() {
        var batchSizeSummary = new SummaryStatistics();
        JugglerRelay sink = new JugglerRelay() {
            @Override
            public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName,
                                                                  List<JugglerEvent> events) {
                batchSizeSummary.addValue(events.size());
                return CompletableFuture.supplyAsync(() -> events.stream()
                        .map(ev -> new EventStatus(200))
                        .collect(Collectors.toList()));
            }

            @Override
            public boolean attemptServe() {
                return true;
            }
        };

        target = new JugglerTarget("iva.juggler.search.yandex.net", TargetState.of(sink, 1),
                metrics,
                new JugglerTargetOptions()
                        .setTimer(timer)
                        .setFlusherExecutor(flusher)
                        .setQueueCapacity(100)
                        .setBatchSizeHint(10)
                        .setFlushIntervalMillis(50));

        for (int i = 0; i < 10; i++) {
            target.addEvent(new JugglerEvent("solomon-" + i, "hugeService", "", JugglerEvent.Status.CRIT, "oops")).join();
        }

        Assert.assertEquals("Avg batch size is too small", batchSizeSummary.getMean(), 1, 1e-6);
    }

    @Test
    public void drainWhenBatchIsReady() {
        var batchSizeSummary = new SummaryStatistics();
        JugglerRelay sink = new JugglerRelay() {
            @Override
            public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName,
                                                                  List<JugglerEvent> events) {
                batchSizeSummary.addValue(events.size());
                return CompletableFuture.supplyAsync(() -> events.stream()
                        .map(ev -> new EventStatus(200))
                        .collect(Collectors.toList()));
            }

            @Override
            public boolean attemptServe() {
                return true;
            }
        };

        target = new JugglerTarget("iva.juggler.search.yandex.net", TargetState.of(sink, 1),
                metrics,
                new JugglerTargetOptions()
                    .setTimer(timer)
                    .setFlusherExecutor(flusher)
                    .setQueueCapacity(100)
                    .setBatchSizeHint(10)
                    .setFlushIntervalMillis(3000));

        var futures = IntStream.range(0, 1000)
                .mapToObj(i -> {
                    if (i % 20 == 0) {
                        try {
                            Thread.sleep(10);
                        } catch (InterruptedException e) {
                            // pass
                        }
                    }
                    return target.addEvent(new JugglerEvent("solomon-" + i, "hugeService", "", JugglerEvent.Status.CRIT, "oops"));
                })
                .collect(Collectors.toList());

        var results = CompletableFutures.allOf(futures).join();

        Int2ObjectMap<MutableInt> counters = new Int2ObjectOpenHashMap<>();
        counters.put(200, new MutableInt(0));
        counters.put(429, new MutableInt(0));
        results.forEach(result -> counters.get(result.code).increment());

        Assert.assertTrue(counters.get(200).intValue() >= 995);
        Assert.assertTrue(counters.get(429).intValue() <= 5);
        Assert.assertTrue("Avg batch size is too small", batchSizeSummary.getMean() > 5);
    }

    @Test
    public void allEventsAreCompletedOnClose() {
        JugglerRelay sink = new JugglerRelay() {
            @Override
            public CompletableFuture<List<EventStatus>> sendBatch(String sourceAppName,
                                                                  List<JugglerEvent> events) {
                return CompletableFuture.supplyAsync(() -> events.stream()
                        .map(ev -> new EventStatus(200))
                        .collect(Collectors.toList()));
            }

            @Override
            public boolean attemptServe() {
                return true;
            }
        };

        target = new JugglerTarget("iva.juggler.search.yandex.net", TargetState.of(sink, 1),
                metrics,
                new JugglerTargetOptions()
                        .setTimer(timer)
                        .setFlusherExecutor(flusher)
                        .setQueueCapacity(100)
                        .setBatchSizeHint(10)
                        .setFlushIntervalMillis(3000));

        new Thread(() -> {
            try {
                Thread.sleep(50);
            } catch (InterruptedException e) {
                //
            }
            target.close();
        }).start();

        var futures = IntStream.range(0, 500).parallel()
                .mapToObj(i -> {
                    if (i % 20 == 0) {
                        try {
                            Thread.sleep(20);
                        } catch (InterruptedException e) {
                            // pass
                        }
                    }
                    return target.addEvent(new JugglerEvent("solomon-" + i, "hugeService", "", JugglerEvent.Status.CRIT, "oops"));
                })
                .collect(Collectors.toList());


        var results = CompletableFutures.allOf(futures)
                .completeOnTimeout(null, 5, TimeUnit.SECONDS)
                .join();
        Assert.assertNotNull(results);

        Int2ObjectMap<MutableInt> counters = new Int2ObjectOpenHashMap<>();
        results.forEach(result -> counters.computeIfAbsent(result.code, ign -> new MutableInt(0)).increment());
        System.out.println(counters);
    }

}
