package ru.yandex.juggler.target;

import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.mutable.MutableInt;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.juggler.relay.CircuitBreakingAware;

import static org.hamcrest.Matchers.closeTo;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class WeightedChoiceTest {
    private Random random;
    private static final int maxRetriesForReady = 20;

    @Before
    public void setUp() {
        random = new Random(42);
    }

    @Test
    public void empty() {
        WeightedChoice<Node> choice = new WeightedChoice<>(List.of(), List.of(), random, maxRetriesForReady);
        Assert.assertTrue(choice.choose().isEmpty());
    }

    @Test
    public void allDead() {
        Node dead = Node.dead("one");
        WeightedChoice<Node> choice = new WeightedChoice<>(List.of(dead), List.of(1), random, maxRetriesForReady);
        Assert.assertTrue(choice.choose().isEmpty());
    }

    @Test
    public void oneDeadOneLives() {
        Node dead = Node.dead("one");
        Node alive = Node.alive("two");
        WeightedChoice<Node> choice = new WeightedChoice<>(List.of(dead, alive), List.of(1, 5), random,
                maxRetriesForReady);
        for (int i = 0; i < 100; i++) {
            Optional<Node> chosen = choice.choose();
            Assert.assertTrue(chosen.isPresent());
            Assert.assertEquals("two", chosen.get().name);
        }
    }

    @Test
    public void twoLivesDifferentWeights() {
        Node one = Node.alive("one");
        Node two = Node.dead("two");
        Node three = Node.alive("three");
        WeightedChoice<Node> choice = new WeightedChoice<>(
                List.of(one, two, three), List.of(1, 5, 5), random, maxRetriesForReady);
        HashMap<String, MutableInt> counters = new HashMap<>();
        for (int i = 0; i < 12000; i++) {
            Optional<Node> chosen = choice.choose();
            Assert.assertTrue(chosen.isPresent());
            counters.computeIfAbsent(chosen.get().name, ignore -> new MutableInt(0)).increment();
        }
        Assert.assertThat((double) counters.get("one").intValue(), closeTo(2000, 100));
        Assert.assertThat((double) counters.get("three").intValue(), closeTo(10000, 100));
    }

    @Test
    public void disabledByWeight() {
        Node one = Node.alive("one");
        Node two = Node.alive("two");
        Node three = Node.alive("three");
        WeightedChoice<Node> choice = new WeightedChoice<>(
                List.of(one, two, three), List.of(0, 1, 0), random, maxRetriesForReady);
        for (int i = 0; i < 100; i++) {
            Optional<Node> chosen = choice.choose();
            Assert.assertTrue(chosen.isPresent());
            Assert.assertEquals("two", chosen.get().name);
        }
    }

    @Test
    public void allDisabledByWeight() {
        Node one = Node.alive("one");
        Node two = Node.alive("two");
        Node three = Node.alive("three");
        WeightedChoice<Node> choice = new WeightedChoice<>(
                List.of(one, two, three), List.of(0, 0, 0), random, maxRetriesForReady);
        for (int i = 0; i < 100; i++) {
            Optional<Node> chosen = choice.choose();
            Assert.assertTrue(chosen.isEmpty());
        }
    }

    @Test
    public void many() {
        List<Node> nodes = IntStream.range(0, 10).mapToObj(i -> Node.alive("N" + i)).collect(Collectors.toList());
        List<Integer> weights = IntStream.range(1, 1 + nodes.size()).boxed().collect(Collectors.toList());
        WeightedChoice<Node> choice = new WeightedChoice<>(nodes, weights, random, maxRetriesForReady);
        HashMap<String, MutableInt> counters = new HashMap<>();
        for (int i = 0; i < 550000; i++) {
            Optional<Node> chosen = choice.choose();
            Assert.assertTrue(chosen.isPresent());
            counters.computeIfAbsent(chosen.get().name, ignore -> new MutableInt(0)).increment();
        }
        for (int i = 0; i < nodes.size(); i++) {
            Assert.assertThat((double) counters.get("N" + i).intValue(), closeTo((i + 1) * 10000, 350));
        }
    }

    private static class Node implements CircuitBreakingAware {
        private final String name;
        private final boolean alive;

        private Node(String name, boolean alive) {
            this.name = name;
            this.alive = alive;
        }

        public static Node dead(String name) {
            return new Node(name, false);
        }

        public static Node alive(String name) {
            return new Node(name, true);
        }

        @Override
        public boolean attemptServe() {
            return alive;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Node node = (Node) o;
            return alive == node.alive &&
                    name.equals(node.name);
        }

        @Override
        public int hashCode() {
            return Objects.hash(name, alive);
        }

        @Override
        public String toString() {
            return "Node{" +
                    "name='" + name + '\'' +
                    ", alive=" + alive +
                    '}';
        }
    }
}
