package ru.yandex.persqueue.examples.read;

import java.util.concurrent.Flow;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.ByteString;
import com.yandex.ydb.core.auth.TokenAuthProvider;

import ru.yandex.persqueue.PersqueueClient;
import ru.yandex.persqueue.codec.Decode;
import ru.yandex.persqueue.read.ReadSession;
import ru.yandex.persqueue.read.event.AbstractEventHandler;
import ru.yandex.persqueue.read.event.Event;
import ru.yandex.persqueue.read.event.Message;
import ru.yandex.persqueue.read.settings.EventHandlersSettings;
import ru.yandex.persqueue.read.settings.ReadSessionSettings;
import ru.yandex.persqueue.settings.GrpcTransportSettings;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadFromTopicSample {

    public static void main(String[] args) throws InterruptedException {
        var authProvider = new TokenAuthProvider(System.getenv("SOLOMON_TOKEN"));

        var client = PersqueueClient.create(GrpcTransportSettings.newBuilder()
                .endpoint("myt.logbroker-prestable.yandex.net:2135")
                .database("/Root")
                .authProvider(authProvider)
                .build());

        var settings = ReadSessionSettings.newBuilder()
                .addTopic("alexnick/test")
                .consumerName("alexnick/gordiychuk")
                .executor(ForkJoinPool.commonPool())
                .eventHandlers(EventHandlersSettings.newBuilder()
                        .executor(ForkJoinPool.commonPool())
                        .commonHandler(new Handler())
                        .build())
                .build();

        ReadSession readSession = client.createReadSession(settings);
        readSession.start();

        TimeUnit.MINUTES.sleep(5L);

        readSession.close();
        client.close();
    }

    private static class Handler extends AbstractEventHandler implements Flow.Subscriber<Event> {
        private Flow.Subscription subscription;

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.subscription = subscription;
            this.subscription.request(1);
        }

        @Override
        public void onNext(Event item) {
            item.visit(this);
            subscription.request(1);
        }

        @Override
        public void onMessage(Message event) {
            System.out.println(event);
            // decode inbound message
            ByteString decoded = Decode.decodeToByteString(event.getCodec(), event.getData());
            // you business logic
            System.out.println(decoded.toStringUtf8());
            // commit changes at the end
            // event.commit();
        }

        @Override
        public void onError(Throwable throwable) {
            // Error that not possible auto resolve by read session
            throwable.printStackTrace();
            System.exit(1);
        }

        @Override
        public void onComplete() {
            // No more new events will be send
            System.exit(0);
        }
    }
}
