package ru.yandex.persqueue.read.impl;

import javax.annotation.concurrent.GuardedBy;

import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.Assigned;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.persqueue.read.PartitionStream;
import ru.yandex.persqueue.read.PartitionStreamKey;
import ru.yandex.persqueue.read.impl.actor.ActorEvents;
import ru.yandex.persqueue.read.impl.actor.ReadSessionActor;

/**
 * @author Vladimir Gordiychuk
 */
public class PartitionStreamImpl implements PartitionStream, AutoCloseable {
    private final Logger logger = LoggerFactory.getLogger(PartitionStreamImpl.class);

    private final ReadSessionActor actor;
    private final PartitionStreamKey key;
    private final long assignId;
    @GuardedBy("this")
    private boolean assignConfirmed;
    @GuardedBy("this")
    private boolean destroyConfirmed;

    private final Long2ObjectMap<Cookie> cookieById = new Long2ObjectOpenHashMap<>();

    @GuardedBy("this")
    private final Long2ObjectMap<Cookie> cookieByUncommittedOffset = new Long2ObjectOpenHashMap<>();

    private volatile boolean closed;

    public PartitionStreamImpl(ReadSessionActor actor, Assigned assign) {
        this.actor = actor;
        this.key = new PartitionStreamKey(assign.getTopic().getPath(), assign.getCluster(), assign.getPartition());
        this.assignId = assign.getAssignId();
    }

    @Override
    public PartitionStreamKey getKey() {
        return key;
    }

    @Override
    public long getAssignId() {
        return assignId;
    }

    @Override
    public String getTopicPath() {
        return key.topic;
    }

    @Override
    public String getClusterName() {
        return key.cluster;
    }

    @Override
    public long getPartition() {
        return key.partition;
    }

    @Override
    public void requestStatus() {
        if (closed) {
            return;
        }

        actor.send(new ActorEvents.RequestPartitionStatus(this));
    }

    @Override
    public synchronized void commit(long startOffset, long endOffset) {
        if (closed) {
            return;
        }

        for (long offset = startOffset; offset < endOffset; offset++) {
            var cookie = cookieByUncommittedOffset.remove(offset);
            if (cookie == null) {
                throw new UnexpectedResultException("Offset " + Long.toUnsignedString(offset) +
                        " was already committed or not read at partitionStream " + this, StatusCode.BAD_REQUEST);
            }

            if (cookie.commitMessage()) {
                actor.send(new ActorEvents.Commit(this, cookie));
            }
        }
    }

    public synchronized void confirmAssign(long readOffset, long commitOffset) {
        if (assignConfirmed) {
            throw new UnexpectedResultException("Assign " + assignId + " for partition " + key + " already confirmed", StatusCode.BAD_REQUEST);
        }

        if (destroyConfirmed) {
            skipConfirm("assign", "already confirmed destroy");
            return;
        }

        if (closed) {
            skipConfirm("assign", "already closed");
            return;
        }

        assignConfirmed = true;
        actor.send(new ActorEvents.ConfirmAssign(this, readOffset, commitOffset));
    }

    public synchronized void confirmDestroy() {
        if (destroyConfirmed) {
            throw new UnexpectedResultException("Destroy " + Long.toUnsignedString(assignId) + " for partition " + key + " already confirmed", StatusCode.BAD_REQUEST);
        }

        if (closed) {
            skipConfirm("destroy", "already closed");
            return;
        }

        destroyConfirmed = true;
        actor.send(new ActorEvents.ConfirmDestroy(this));
    }

    public void addCookie(long id, long startOffset, long endOffset) {
        Cookie cookie = new Cookie(id, startOffset, endOffset);
        synchronized (this) {
            var prev = cookieById.put(cookie.id, cookie);
            if (prev != null) {
                throw new UnexpectedResultException(
                        "PartitionStream " + this + " already contains cookie " + Long.toUnsignedString(cookie.id),
                        StatusCode.INTERNAL_ERROR);
            }
            for (long offset = startOffset; offset < endOffset; offset++) {
                cookieByUncommittedOffset.put(offset, cookie);
            }
        }
    }

    public Cookie committed(long cookieId) {
        return cookieById.remove(cookieId);
    }

    private void skipConfirm(String op, String reason) {
        logger.info("Skip confirm {} {} for partition {} because {}", op, Long.toUnsignedString(assignId), key, reason);
    }

    @Override
    public void close() {
        closed = true;
    }

    @Override
    public String toString() {
        return "PartitionStream{" +
                "key=" + key +
                ", assignId=" + Long.toUnsignedString(assignId) +
                '}';
    }
}
