package ru.yandex.persqueue.read.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.Iterables;
import com.google.protobuf.Empty;
import com.google.protobuf.TextFormat;
import com.yandex.ydb.core.Issue;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.persqueue.cluster_discovery.YdbPersqueueClusterDiscovery.ClusterInfo;
import com.yandex.ydb.persqueue.cluster_discovery.YdbPersqueueClusterDiscovery.DiscoverClustersRequest;
import com.yandex.ydb.persqueue.cluster_discovery.YdbPersqueueClusterDiscovery.DiscoverClustersResult;
import com.yandex.ydb.persqueue.cluster_discovery.YdbPersqueueClusterDiscovery.ReadSessionParams;
import io.grpc.SynchronizationContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.persqueue.read.ReadSession;
import ru.yandex.persqueue.read.impl.actor.ActorEvents;
import ru.yandex.persqueue.read.impl.actor.ActorEvents.Connect;
import ru.yandex.persqueue.read.impl.actor.ActorEvents.MemoryChunkConsumed;
import ru.yandex.persqueue.read.impl.actor.EventProducer;
import ru.yandex.persqueue.read.impl.actor.EventPublisher;
import ru.yandex.persqueue.read.impl.actor.ReadSessionActor;
import ru.yandex.persqueue.read.impl.actor.ReadSessionActorImpl;
import ru.yandex.persqueue.read.impl.actor.ReadSessionRetryContext;
import ru.yandex.persqueue.read.settings.ReadSessionSettings;
import ru.yandex.persqueue.read.settings.TopicReadSettings;
import ru.yandex.persqueue.rpc.PqRpc;
import ru.yandex.persqueue.rpc.RpcPool;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadSessionImpl implements EventProducer, ReadSession {
    private static final Logger logger = LoggerFactory.getLogger(ReadSessionImpl.class);

    private final String endpoint;
    private final ReadSessionSettings settings;
    private final RpcPool rpcPool;
    private final SynchronizationContext context;

    private final Map<String, ReadSessionActor> clusterReadSessionByName = new HashMap<>();
    private final ReadSessionRetryContext retryContext;
    private final EventPublisher publisher;
    private boolean started;
    private boolean closed;
    private PqRpc rpc;

    public ReadSessionImpl(String endpoint, RpcPool rpcPool, ReadSessionSettings settings) {
        this.endpoint = endpoint;
        this.settings = settings;
        this.rpcPool = rpcPool;
        this.retryContext = new ReadSessionRetryContext(settings.retry, this::start);
        this.context = new SynchronizationContext((t, e) -> {
            onError(Status.of(StatusCode.CLIENT_INTERNAL_ERROR, Issue.of(Throwables.getStackTraceAsString(e), Issue.Severity.ERROR)));
        });
        this.publisher = new EventPublisher(this, settings.executor, ReadSessionActorImpl.MAX_BATCH_SIZE, settings.handler.commonEvents);
    }

    @Override
    public void start() {
        context.execute(() -> {
            if (started || closed) {
                return;
            }

            if (rpc == null) {
                rpc = rpcPool.getRpc(endpoint);
            }

            rpc.discoverClusters(discoveryRequest())
                    .whenComplete((result, e) -> {
                        if (e != null) {
                            result = Result.error(e);
                        }

                        if (result.isSuccess()) {
                            onClustersDiscovered(result.expect("discoverClusters()"));
                        } else {
                            logger.warn("Cluster discovery {} request failed: {}", endpoint, result);
                            if (result.getCode() == StatusCode.CLIENT_CALL_UNIMPLEMENTED) {
                                var actor = new ReadSessionActorImpl("", endpoint, settings, publisher, rpcPool);
                                clusterReadSessionByName.put("", actor);
                                actor.send(new Connect());
                                started = true;
                                rpc.close();
                                rpc = null;
                                retryContext.success();
                                return;
                            }

                            onError(result.toStatus());
                        }
                    });
        });
    }

    private void onError(Status status) {
        context.execute(() -> {
            if (closed) {
                return;
            }

            if (!retryContext.scheduleRetry(status.getCode())) {
                abort(status);
            }
        });
    }

    private void abort(Status status) {
        context.execute(() -> {
            if (closed) {
                return;
            }

            publisher.closeExceptionally(status);
            close();
        });
    }

    private void onClustersDiscovered(DiscoverClustersResult result) {
        context.execute(() -> {
            if (started || closed) {
                return;
            }

            if (result.getReadSessionsClustersCount() != settings.topics.size()) {
                String msg = "Unexpected reply from cluster discovery. Sizes of topics arrays don't match: " +
                        result.getReadSessionsClustersCount() + " vs " + settings.topics.size();
                onError(Status.of(StatusCode.INTERNAL_ERROR, Issue.of(msg, Issue.Severity.ERROR)));
                return;
            }

            Map<String, ClusterSetup> setupByName = new HashMap<>();
            List<Issue> issues = new ArrayList<>();
            StatusCode errorStatus = StatusCode.INTERNAL_ERROR;
            for (int index = 0; index < settings.topics.size(); index++) {
                var topic = settings.topics.get(index);
                var clusters = result.getReadSessionsClusters(index);
                for (var cluster : clusters.getClustersList()) {
                    String name = cluster.getName().toLowerCase();
                    if (!settings.clusters.isEmpty() && !settings.clusters.contains(name)) {
                        continue;
                    }

                    var setup = setupByName.computeIfAbsent(name, ignore -> new ClusterSetup(cluster));
                    if (Strings.isNullOrEmpty(setup.endpoint)) {
                        String msg = "Unexpected reply from cluster discovery. Empty endpoint for cluster " + TextFormat.shortDebugString(cluster);
                        issues.add(Issue.of(msg, Issue.Severity.ERROR));
                    }

                    if (!Objects.equals(setup.endpoint, cluster.getEndpoint())) {
                        String msg = "Unexpected reply from cluster discovery. Different endpoints for one cluster name. Cluster: "
                                + setup.name + ". \"" + setup.endpoint + "\" vs \""
                                + cluster.getEndpoint() + "\"";
                        issues.add(Issue.of(msg, Issue.Severity.ERROR));
                    }

                    setup.topics.add(topic);
                }
            }

            if (!issues.isEmpty()) {
                abort(Status.of(StatusCode.INTERNAL_ERROR, issues.toArray(Issue[]::new)));
                return;
            }

            for (String cluster : settings.clusters) {
                if (setupByName.get(cluster) == null) {
                    errorStatus = StatusCode.BAD_REQUEST;
                    issues.add(Issue.of("Unsupported cluster: " + cluster, Issue.Severity.ERROR));
                }
            }

            if (!issues.isEmpty()) {
                abort(Status.of(errorStatus, issues.toArray(Issue[]::new)));
                return;
            }

            int maxMemoryUsage = settings.maxMemoryUsageBytes / setupByName.size();
            for (var setup : setupByName.values()) {
                var clusterSettings= this.settings.toBuilder()
                        .setTopics(setup.topics)
                        .maxMemoryUsage(maxMemoryUsage)
                        .build();

                var actor = new ReadSessionActorImpl(setup.name, setup.endpoint, clusterSettings, publisher, rpcPool);
                clusterReadSessionByName.put(setup.name, actor);
                actor.send(new Connect());
            }

            started = true;
            rpc.close();
            rpc = null;
            retryContext.success();
        });
    }

    @Override
    public void close() {
        context.execute(() -> {
            if (closed) {
                return;
            }

            closed = true;
            publisher.close();
            retryContext.close();
            clusterReadSessionByName.values().forEach(actor -> actor.send(new ActorEvents.Disconnect()));
            clusterReadSessionByName.clear();
        });
    }

    private DiscoverClustersRequest discoveryRequest() {
        DiscoverClustersRequest.Builder req = DiscoverClustersRequest.newBuilder();
        for (var topic : settings.topics) {
            ReadSessionParams.Builder params = ReadSessionParams.newBuilder();
            params.setTopic(topic.path);
            if (settings.readOriginal) {
                params.setAllOriginal(Empty.newBuilder().build());
            } else {
                params.setMirrorToCluster(Iterables.getOnlyElement(settings.clusters));
            }
            req.addReadSessions(params);
        }
        return req.build();
    }

    @Override
    public void chunkConsumed(String cluster) {
        context.execute(() -> {
            var actor = clusterReadSessionByName.get(cluster);
            if (actor != null) {
                actor.send(MemoryChunkConsumed.INSTANCE);
            }
        });
    }

    @Override
    public void cancel() {
        context.execute(this::close);
    }

    private static class ClusterSetup {
        private final String name;
        private final String endpoint;
        private final boolean available;
        private final List<TopicReadSettings> topics = new ArrayList<>();

        public ClusterSetup(ClusterInfo info) {
            this.name = info.getName().toLowerCase();
            this.endpoint = info.getEndpoint();
            this.available = info.getAvailable();
        }
    }

}
