package ru.yandex.persqueue.read.impl.actor;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.core.rpc.OutStreamObserver;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.persqueue.read.impl.protocol.ClientRequestProto;
import ru.yandex.persqueue.read.impl.protocol.ServerResponseProto;
import ru.yandex.persqueue.read.impl.protocol.handler.TransportEventHandler;
import ru.yandex.persqueue.rpc.PqRpc;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ClientServerSession implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ClientServerSession.class);

    private final String cluster;
    private final TransportEventHandler handler;
    private final EventQueue<MigrationStreamingReadServerMessage> inbound;
    private final OutStreamObserver<MigrationStreamingReadClientMessage> server;

    private String sessionId = "";

    public ClientServerSession(String cluster, PqRpc rpc, Runnable onEventCallback, TransportEventHandler handler) {
        this.cluster = cluster;
        this.handler = handler;
        this.inbound = new EventQueueImpl<>(onEventCallback);
        this.server = rpc.readSessionV1(new ServerResponseObserver(inbound));
    }

    public void send(MigrationStreamingReadClientMessage message) {
        logSendMessage(message);
        server.onNext(message);
    }

    /**
     * Abort active session with send error on server
     */
    public void abort(com.yandex.ydb.core.Status status) {
        logger.info("{}:{} <- {}", cluster, sessionId, status);
        server.onError(new UnexpectedResultException("client error", status.getCode(), status.getIssues()));
    }

    /**
     * Close active session
     */
    @Override
    public void close() {
        logger.info("{}:{} <- complete", cluster, sessionId);
        server.onCompleted();
    }

    public void setSessionId(String sessionId) {
        this.sessionId = sessionId;
    }

    public void processInbound() {
        try {
            var error = inbound.getError();
            if (error != null) {
                logger.warn("{}:{} -> {}", cluster, sessionId, error);
                handler.onError(error);
                return;
            }

            if (inbound.isCompleted()) {
                logger.info("{}:{} -> complete", cluster, sessionId);
                handler.onComplete();
                return;
            }

            MigrationStreamingReadServerMessage message;
            while ((message = inbound.dequeue()) != null) {
                logReceiveMessage(message);
                if (!ServerResponseProto.dispatch(message, handler)) {
                    break;
                }
            }
        } catch (Throwable e) {
            logger.error("{}:{} client side error", cluster, sessionId, e);
            handler.onError(ClientRequestProto.toStatus(e));
        }
    }

    @Override
    public String toString() {
        return cluster + ":" + sessionId;
    }

    private void logSendMessage(MigrationStreamingReadClientMessage message) {
        switch (message.getRequestCase()) {
            case INIT_REQUEST:
            case START_READ:
            case RELEASED:
            case STATUS:
                if (logger.isInfoEnabled()) {
                    logger.info("{}:{} <- {}", cluster, sessionId, ClientRequestProto.toString(message));
                }
                break;
            default:
                if (logger.isDebugEnabled()) {
                    logger.debug("{}:{} <- {}", cluster, sessionId, ClientRequestProto.toString(message));
                }
        }
    }

    private void logReceiveMessage(MigrationStreamingReadServerMessage message) {
        switch (message.getResponseCase()) {
            case ASSIGNED:
            case RELEASE:
            case PARTITION_STATUS:
            case INIT_RESPONSE:
                if (logger.isInfoEnabled()) {
                    logger.info("{}:{} -> {}", cluster, sessionId, ServerResponseProto.toString(message));
                }
            default:
                if (logger.isDebugEnabled()) {
                    logger.debug("{}:{} -> {}", cluster, sessionId, ServerResponseProto.toString(message));
                }
        }
    }
}
