package ru.yandex.persqueue.read.impl.actor;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Flow;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.UnexpectedResultException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.persqueue.read.event.Event;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Vladimir Gordiychuk
 */
public class EventPublisher implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(EventPublisher.class);

    private final EventProducer producer;
    private final ActorRunner actor;
    private final EventQueue<Event> events;
    private final Subscription subscription;
    private final int memoryChunkSize;
    private final ConcurrentMap<String, MemoryUse> memoryUseByCluster = new ConcurrentHashMap<>();
    private final Flow.Subscriber<? super Event> subscriber;

    public EventPublisher(EventProducer producer, Executor executor, int memoryChunkSize, Flow.Subscriber<? super Event> subscriber) {
        this.producer = producer;
        this.actor = new ActorRunner(this::act, executor);
        this.events = new EventQueueImpl<>(actor::schedule);
        this.subscription = new Subscription();
        this.subscriber = subscriber;
        this.memoryChunkSize = memoryChunkSize;
        subscriber.onSubscribe(subscription);
    }

    public void submit(Event event) {
        memoryUse(event.getPartitionStream().getClusterName()).addMemory(event.memoryUseBytes());
        events.enqueue(event);
    }

    public int getMemoryUseBytes(String cluster) {
        return memoryUse(cluster).memoryUseBytes.get();
    }

    public void closeExceptionally(Status status) {
        events.onError(status);
    }

    private void act() {
        if (subscription.closed) {
            Event event;
            while ((event = events.dequeue()) != null) {
                consumed(event);
            }
        }

        if (events.isDone()) {
            var error = events.getError();
            if (error != null) {
                consumeError(error);
            } else {
                consumeComplete();
            }
        }

        consumeDemand();
    }

    private void consumeDemand() {
        if (subscription.demand.get() == 0) {
            return;
        }

        Event event;
        while ((event = events.dequeue()) != null) {
            if (!consumeNext(event)) {
                return;
            }

            if (subscription.demand.decrementAndGet() == 0) {
                return;
            }
        }
    }

    private boolean consumeNext(Event event) {
        try {
            consumed(event);
            subscriber.onNext(event);
            return true;
        } catch (Throwable ex) {
            consumeError(ex);
            return false;
        }
    }

    private void consumeError(Status error) {
        consumeError(new UnexpectedResultException("ReadSession failed", error.getCode(), error.getIssues()));
    }

    private void consumeError(Throwable error) {
        if (subscription.closed) {
            return;
        }

        try {
            subscriber.onError(error);
        } catch (Throwable e) {
            logger.warn("Failed consume error {}", error, e);
        } finally {
            subscription.cancel();
        }
    }

    private void consumeComplete() {
        if (subscription.closed) {
            return;
        }

        try {
            subscriber.onComplete();
        } catch (Throwable e) {
            logger.warn("Failed consume complete", e);
        } finally {
            subscription.cancel();
        }
    }

    @Override
    public void close() {
        events.onComplete();
    }

    private void consumed(Event event) {
        var cluster = event.getPartitionStream().getClusterName();
        var memoryUse = memoryUseByCluster.computeIfAbsent(cluster, MemoryUse::new);
        memoryUse.addConsumedMemoryBytes(event.memoryUseBytes());
    }

    private MemoryUse memoryUse(String cluster) {
        return memoryUseByCluster.computeIfAbsent(cluster, MemoryUse::new);
    }

    private class Subscription implements Flow.Subscription {
        private final AtomicLong demand = new AtomicLong();
        private volatile boolean closed;

        @Override
        public void request(long n) {
            checkArgument(n > 0, "Non possible demand");
            long p, d;
            do {
                p = demand.get();
                d = p + n;
            } while (!demand.compareAndSet(p, d < p ? Long.MAX_VALUE : d));
            actor.schedule();
        }

        @Override
        public void cancel() {
            closed = true;
            producer.cancel();
            events.onComplete();
            actor.schedule();
        }
    }

    private class MemoryUse {
        private final String cluster;
        private final AtomicInteger memoryUseBytes = new AtomicInteger();

        private int memoryConsumeBytes;
        private int nextConsumedThreshold;

        public MemoryUse(String cluster) {
            this.cluster = cluster;
            this.nextConsumedThreshold = memoryChunkSize;
        }

        private void addMemory(int bytes) {
            memoryUseBytes.addAndGet(bytes);
        }

        private void addConsumedMemoryBytes(int bytes) {
            memoryConsumeBytes += bytes;
            memoryUseBytes.addAndGet(-bytes);
            if (memoryConsumeBytes >= nextConsumedThreshold) {
                nextConsumedThreshold += memoryChunkSize;
                producer.chunkConsumed(cluster);
            }
        }
    }
}
