package ru.yandex.persqueue.read.impl.actor;

import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;

import com.yandex.ydb.core.Status;

/**
 * @author Vladimir Gordiychuk
 */
public class EventQueueImpl<T> implements EventQueue<T> {
    private static final Object NIL = new Object();
    private static final Object COMPLETE = new Object();

    private final Runnable incomeEventCallback;

    private final ConcurrentLinkedDeque<T> events = new ConcurrentLinkedDeque<>();
    private final AtomicReference<Object> done = new AtomicReference<>(NIL);

    public EventQueueImpl(Runnable incomeEventCallback) {
        this.incomeEventCallback = incomeEventCallback;
    }

    @Override
    public void enqueue(T event) {
        // race at this place not a problem
        if (isDone()) {
            return;
        }

        events.add(event);
        incomeEventCallback.run();
    }

    @Override
    public void onError(Status status) {
        if (done.compareAndSet(NIL, status)) {
            incomeEventCallback.run();
        }
    }

    @Override
    public void onComplete() {
        if (done.compareAndSet(NIL, COMPLETE)) {
            incomeEventCallback.run();
        }
    }

    @Nullable
    @Override
    public T dequeue() {
        return events.poll();
    }

    @Nullable
    @Override
    public Status getError() {
        var result = done.get();
        if (result == NIL || result == COMPLETE) {
            return null;
        }
        return (Status) result;
    }

    @Override
    public boolean isCompleted() {
        return done.get() == COMPLETE;
    }

    @Override
    public boolean isDone() {
        return done.get() != NIL;
    }
}
