package ru.yandex.persqueue.read.impl.actor;

import java.util.EnumSet;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.utils.Async;
import io.netty.util.Timeout;

import ru.yandex.persqueue.read.settings.RetrySettings;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadSessionRetryContext implements AutoCloseable {
    private static final EnumSet<StatusCode> UNRETRYABLE_STATUSES = EnumSet.of(
            StatusCode.UNUSED_STATUS,
            StatusCode.BAD_REQUEST,
            StatusCode.UNAUTHORIZED,
            StatusCode.PRECONDITION_FAILED,
            StatusCode.UNSUPPORTED,
            StatusCode.ALREADY_EXISTS,
            StatusCode.NOT_FOUND,
            StatusCode.CLIENT_UNAUTHENTICATED,
            StatusCode.CLIENT_CALL_UNIMPLEMENTED
    );

    private final RetrySettings settings;
    private final Runnable runnable;
    private int retryNumber;
    private Timeout scheduledTimeout;

    public ReadSessionRetryContext(RetrySettings settings, Runnable runnable) {
        this.settings = settings;
        this.runnable = runnable;
    }

    public boolean scheduleRetry(StatusCode code) {
        if (UNRETRYABLE_STATUSES.contains(code)) {
            return false;
        }

        int retry = retryNumber++;
        if (retry >= settings.maxRetries) {
            return false;
        }

        scheduleNext(backoffTimeMillis(code, retry));
        return true;
    }

    public void success() {
        retryNumber = 0;
        if (scheduledTimeout != null) {
            scheduledTimeout.cancel();
        }
    }

    private void scheduleNext(long delayMillis) {
        var prev = scheduledTimeout;
        if (prev != null) {
            prev.cancel();
        }

        scheduledTimeout = Async.runAfter(this::runScheduled, delayMillis, TimeUnit.MILLISECONDS);
    }

    private void runScheduled(Timeout timeout) {
        if (timeout.isCancelled()) {
            return;
        }

        runnable.run();
    }

    private long backoffTimeMillis(int retryNumber) {
        int slots = 1 << Math.min(retryNumber, settings.backoffCeiling);
        return randomizeDelay(settings.backoffSlotMillis * slots);
    }

    private long randomizeDelay(long delayMillis) {
        if (delayMillis == 0) {
            return 0;
        }

        var half = delayMillis / 2;
        return half + ThreadLocalRandom.current().nextLong(half);
    }

    private long backoffTimeMillis(StatusCode code, int retryNumber) {
        // TODO: fast retry some statuses first time? (gordiychuk@)
        return backoffTimeMillis(retryNumber);
    }

    @Override
    public synchronized void close() {
        if (scheduledTimeout != null) {
            scheduledTimeout.cancel();
        }
    }
}
