package ru.yandex.persqueue.read.impl.event;

import java.time.Instant;
import java.util.Map;

import com.google.common.base.Joiner;
import com.google.protobuf.ByteString;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.DataBatch.MessageData;

import ru.yandex.persqueue.codec.Codec;
import ru.yandex.persqueue.read.PartitionStream;
import ru.yandex.persqueue.read.event.EventHandler;
import ru.yandex.persqueue.read.event.Message;
import ru.yandex.persqueue.read.impl.PartitionStreamImpl;
import ru.yandex.persqueue.read.impl.protocol.ServerResponseProtoConverter;

/**
 * @author Vladimir Gordiychuk
 */
public class MessageImpl implements Message {
    private final ByteString data;
    private final Codec codec;
    private final long offset;
    private final long seqNo;
    private final long createTimeMillis;
    private final Map<String, String> extraFields;
    private final PartitionStreamImpl partitionStream;

    public MessageImpl(PartitionStreamImpl partitionStream, Map<String, String> extraFields, MessageData data) {
        this.partitionStream = partitionStream;
        this.data = data.getData();
        this.codec = ServerResponseProtoConverter.toCodec(data.getCodec());
        this.offset = data.getOffset();
        this.seqNo = data.getSeqNo();
        this.createTimeMillis = data.getCreateTimestampMs();
        this.extraFields = extraFields;
    }

    @Override
    public ByteString getData() {
        return data;
    }

    @Override
    public Codec getCodec() {
        return codec;
    }

    @Override
    public long getOffset() {
        return offset;
    }

    @Override
    public long getSeqNo() {
        return seqNo;
    }

    @Override
    public long getCreateTimeMillis() {
        return createTimeMillis;
    }

    @Override
    public Map<String, String> getExtraFields() {
        return extraFields;
    }

    @Override
    public void commit() {
        partitionStream.commit(offset);
    }

    @Override
    public PartitionStream getPartitionStream() {
        return partitionStream;
    }

    @Override
    public int memoryUseBytes() {
        return data.size();
    }

    @Override
    public void visit(EventHandler handler) {
        handler.onMessage(this);
    }

    @Override
    public String toString() {
        return "Message{partition=" + partitionStream +
                ", data=" + data.size() + " bytes" +
                ", codec=" + codec +
                ", offset=" + Long.toUnsignedString(offset) +
                ", seqNo=" + Long.toUnsignedString(seqNo) +
                ", createTime=" + Instant.ofEpochMilli(createTimeMillis) +
                ", extraFields={" + Joiner.on(", ").withKeyValueSeparator(": ").join(extraFields) + "}";
    }
}
