package ru.yandex.persqueue.read.impl.protocol;

import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.protobuf.ByteString;
import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import com.yandex.ydb.core.Issue;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage.Commit;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage.InitRequest;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage.Read;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage.Released;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage.StartRead;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage.Status;


/**
 * @author Vladimir Gordiychuk
 */
public class ClientRequestProto {

    /**
     * Handshake request.
     */
    public static MigrationStreamingReadClientMessage init(InitRequest init, String token) {
        return success(token).setInitRequest(init).build();
    }

    /**
     * Request for data.
     */
    public static MigrationStreamingReadClientMessage read(Read read, String token) {
        return success(token).setRead(read).build();
    }

    /**
     * Signal for server that client is ready to get data from partition.
     */
    public static MigrationStreamingReadClientMessage startRead(StartRead startRead, String token) {
        return success(token).setStartRead(startRead).build();
    }

    /**
     * Request for commit of some read data.
     */
    public static MigrationStreamingReadClientMessage commit(Commit commit, String token) {
        return success(token).setCommit(commit).build();
    }

    /**
     * Signal for server that client finished working with partition.
     * Must be sent only after corresponding Release request from server.
     */
    public static MigrationStreamingReadClientMessage release(Released release, String token) {
        return success(token).setReleased(release).build();
    }

    /**
     * Request status of particular partition
     */
    public static MigrationStreamingReadClientMessage partitionStatus(Status partitionStatus, String token) {
        return success(token).setStatus(partitionStatus).build();
    }

    public static Message getRequest(MigrationStreamingReadClientMessage message) {
        switch (message.getRequestCase()) {
            case READ:
                return message.getRead();
            case COMMIT:
                return message.getCommit();
            case STATUS:
                return message.getStatus();
            case RELEASED:
                return message.getReleased();
            case INIT_REQUEST:
                return message.getInitRequest();
            case START_READ:
                return message.getStartRead();
            default:
                throw new UnsupportedOperationException("Unexpected message type from client: " + message.getRequestCase());
        }
    }

    public static com.yandex.ydb.core.Status toStatus(Throwable throwable) {
        var issue = Issue.of(Throwables.getStackTraceAsString(throwable), Issue.Severity.ERROR);
        return com.yandex.ydb.core.Status.of(StatusCode.INTERNAL_ERROR, issue);
    }

    public static String toString(MigrationStreamingReadClientMessage message) {
        if (!message.getToken().isEmpty()) {
            message = message.toBuilder()
              .setToken(ByteString.copyFromUtf8("<TOKEN>"))
              .build();
        }

        return TextFormat.shortDebugString(message);
    }

    private static MigrationStreamingReadClientMessage.Builder success(String token) {
        var builder = MigrationStreamingReadClientMessage.newBuilder();
        if (!Strings.isNullOrEmpty(token)) {
            return builder.setToken(ByteString.copyFromUtf8(token));
        }

        return builder;
    }
}
