package ru.yandex.persqueue.read.impl.protocol;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import com.yandex.ydb.StatusCodesProtos;
import com.yandex.ydb.YdbIssueMessage;
import com.yandex.ydb.core.Issue;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.Assigned;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.Committed;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.DataBatch;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.InitResponse;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.PartitionStatus;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.Release;

import ru.yandex.persqueue.read.impl.protocol.handler.TransportEventHandler;

/**
 * @author Vladimir Gordiychuk
 */
public class ServerResponseProto {
    public static MigrationStreamingReadServerMessage init(InitResponse init) {
        return success().setInitResponse(init).build();
    }

    public static MigrationStreamingReadServerMessage dataBatch(DataBatch dataBatch) {
        return success().setDataBatch(dataBatch).build();
    }

    public static MigrationStreamingReadServerMessage assign(Assigned assign) {
        return success().setAssigned(assign).build();
    }

    public static MigrationStreamingReadServerMessage partitionStatus(PartitionStatus partitionStatus) {
        return success().setPartitionStatus(partitionStatus).build();
    }

    public static MigrationStreamingReadServerMessage commit(Committed commit) {
        return success().setCommitted(commit).build();
    }

    public static MigrationStreamingReadServerMessage release(Release release) {
        return success().setRelease(release).build();
    }

    public static MigrationStreamingReadServerMessage failed(Status status) {
        return MigrationStreamingReadServerMessage.newBuilder()
                .setStatusValue(status.getCode().getCode())
                .addAllIssues(Stream.of(status.getIssues())
                        .map(ServerResponseProto::issueToProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public static boolean isSuccess(MigrationStreamingReadServerMessage message) {
        return message.getStatus() == StatusCodesProtos.StatusIds.StatusCode.SUCCESS
                // TODO: remove this hack when server will set status in each response (gordiychuk@)
                || message.getStatus() == StatusCodesProtos.StatusIds.StatusCode.STATUS_CODE_UNSPECIFIED;
    }

    public static Status getStatus(MigrationStreamingReadServerMessage message) {
        if (isSuccess(message)) {
            return Status.SUCCESS;
        }

        var code = StatusCode.fromProto(message.getStatus());
        Issue[] issues = Issue.fromPb(message.getIssuesList());
        return Status.of(code, issues);
    }



    public static Message getResponse(MigrationStreamingReadServerMessage message) {
        switch (message.getResponseCase()) {
            case INIT_RESPONSE:
                return message.getInitResponse();
            case ASSIGNED:
                return message.getAssigned();
            case DATA_BATCH:
                return message.getDataBatch();
            case RELEASE:
                return message.getRelease();
            case COMMITTED:
                return message.getCommitted();
            case PARTITION_STATUS:
                return message.getPartitionStatus();
            default:
                throw new UnsupportedOperationException("Unexpected message type from server: " + message.getResponseCase());
        }
    }

    public static boolean dispatch(MigrationStreamingReadServerMessage message, TransportEventHandler handler) {
        if (!isSuccess(message)) {
            handler.onError(getStatus(message));
            return false;
        }

        switch (message.getResponseCase()) {
            case INIT_RESPONSE:
                return handler.onInit(message.getInitResponse());
            case ASSIGNED:
                return handler.onAssign(message.getAssigned());
            case DATA_BATCH:
                return handler.onDataBatch(message.getDataBatch());
            case RELEASE:
                return handler.onRelease(message.getRelease());
            case COMMITTED:
                return handler.onCommit(message.getCommitted());
            case PARTITION_STATUS:
                return handler.onPartitionStatus(message.getPartitionStatus());
            default:
                throw new UnsupportedOperationException("Unexpected message type from server: " + message.getResponseCase());
        }
    }


    public static String toString(MigrationStreamingReadServerMessage message) {
        return TextFormat.shortDebugString(message);
    }

    private static YdbIssueMessage.IssueMessage issueToProto(Issue issue) {
        return YdbIssueMessage.IssueMessage.newBuilder()
                .setPosition(YdbIssueMessage.IssueMessage.Position.newBuilder()
                        .setColumn(issue.getPosition().getColumn())
                        .setRow(issue.getPosition().getRow())
                        .setFile(issue.getPosition().getFile())
                        .build())
                .setEndPosition(YdbIssueMessage.IssueMessage.Position.newBuilder()
                        .setColumn(issue.getEndPosition().getColumn())
                        .setRow(issue.getEndPosition().getRow())
                        .setFile(issue.getEndPosition().getFile())
                        .build())
                .setSeverity(issue.getSeverity().getCode())
                .setMessage(issue.getMessage())
                .addAllIssues(Stream.of(issue.getIssues())
                        .map(ServerResponseProto::issueToProto)
                        .collect(Collectors.toList()))
                .build();
    }

    private static MigrationStreamingReadServerMessage.Builder success() {
        return MigrationStreamingReadServerMessage.newBuilder()
                .setStatus(StatusCodesProtos.StatusIds.StatusCode.SUCCESS);
    }

}
