package ru.yandex.persqueue.read.settings;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executor;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import com.google.common.util.concurrent.MoreExecutors;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ReadSessionSettings {
    public final List<TopicReadSettings> topics;
    public final String consumerName;
    public final int maxMemoryUsageBytes;
    public final boolean readOriginal;
    public final Set<String> clusters;
    public final Instant startFrom;
    public final Duration maxTimeLag;
    public final Executor executor;
    public final EventHandlersSettings handler;
    public final RetrySettings retry;

    public ReadSessionSettings(Builder builder) {
        this.topics = List.copyOf(builder.topics);
        checkArgument(!this.topics.isEmpty(), "Empty topics list.");
        this.consumerName = requireNonNull(builder.consumerName, "No consumer specified");
        this.readOriginal = builder.readOriginal;
        this.clusters = builder.clusters;
        this.startFrom = builder.startFrom;
        this.maxMemoryUsageBytes = builder.maxMemoryUsageBytes;
        this.maxTimeLag = builder.maxTimeLag;
        this.executor = builder.executor;
        this.handler = requireNonNull(builder.handler);
        this.retry = requireNonNull(builder.retrySettings);
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private String consumerName;
        private final List<TopicReadSettings> topics = new ArrayList<>();
        private int maxMemoryUsageBytes = 100 << 20; // 100 Mib
        private boolean readOriginal = true;
        private Set<String> clusters = Set.of();
        private RetrySettings retrySettings = RetrySettings.DEFAULT;

        private Instant startFrom = Instant.EPOCH;
        private Duration maxTimeLag = Duration.ZERO;
        private Executor executor = MoreExecutors.directExecutor();
        private EventHandlersSettings handler;

        private Builder() {
        }

        private Builder(ReadSessionSettings settings) {
            this.consumerName = settings.consumerName;
            this.topics.addAll(settings.topics);
            this.maxMemoryUsageBytes = settings.maxMemoryUsageBytes;
            this.clusters = settings.clusters;
            this.readOriginal = settings.readOriginal;
            this.retrySettings = settings.retry;
            this.startFrom = settings.startFrom;
            this.maxTimeLag = settings.maxTimeLag;
            this.executor = settings.executor;
            this.handler = settings.handler;
        }

        /**
         * Consumer
         */
        public Builder consumerName(String name) {
            this.consumerName = name;
            return this;
        }

        /**
         * Topic
         */
        public Builder addTopic(TopicReadSettings settings) {
            this.topics.add(settings);
            return this;
        }

        public Builder setTopics(List<TopicReadSettings> topics) {
            this.topics.clear();
            this.topics.addAll(topics);
            return this;
        }

        /**
         * Topic
         * @param path of topic to read
         */
        public Builder addTopic(String path) {
            return addTopic(TopicReadSettings.newBuilder().path(path).build());
        }

        /**
         * Default variant.
         * Read topic instance specified in "Topics" from all clusters.
         */
        public Builder ReadAll() {
            clusters = Set.of();
            readOriginal = true;
            return this;
        }

        /**
         * Read original topic instances specified in "Topics" from several clusters.
         */
        public Builder readOriginal(List<String> clusters) {
            this.clusters = Set.copyOf(clusters);
            this.readOriginal = true;
            return this;
        }

        /**
         * Read mirrored topics specified in {@link Builder#addTopic(TopicReadSettings)} from one cluster.
         */
        public Builder readMirrored(String cluster) {
            clusters = Set.of(requireNonNull(Strings.emptyToNull(cluster), "cluster can't be empty"));
            readOriginal = false;
            return this;
        }

        /**
         * Maximum memory usage for read session.
         * Default 100Mib
         */
        public Builder maxMemoryUsage(int bytes) {
            checkArgument(bytes > 0, "Too small max memory usage. Valid values start from 1 megabyte.");
            this.maxMemoryUsageBytes = bytes;
            return this;
        }

        /**
         * Max message time lag. All messages older that now - MaxTimeLag will be ignored.
         */
        public Builder maxTimeLag(Duration maxLag) {
            this.maxTimeLag = requireNonNull(maxLag);
            return this;
        }

        public Builder startingMessageTimestamp(Instant time) {
            this.startFrom = requireNonNull(time);
            return this;
        }

        public Builder eventHandlers(EventHandlersSettings handlers) {
            this.handler = handlers;
            return this;
        }

        public Builder retrySettings(RetrySettings retrySettings) {
            this.retrySettings = requireNonNull(retrySettings);
            return this;
        }

        /**
         * Executor that will be use to server incoming messages
         */
        public Builder executor(Executor executor) {
            this.executor = requireNonNull(executor);
            return this;
        }

        public ReadSessionSettings build() {
            return new ReadSessionSettings(this);
        }
    }
}
