package ru.yandex.persqueue.read.settings;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Vladimir Gordiychuk
 */
public class RetrySettings {
    public static final RetrySettings DEFAULT = RetrySettings.newBuilder().build();

    public final int maxRetries;
    public final long backoffSlotMillis;
    public final int backoffCeiling;

    public RetrySettings(Builder builder) {
        this.maxRetries = builder.maxRetries;
        this.backoffCeiling = builder.backoffCeiling;
        this.backoffSlotMillis = builder.backoffSlotMillis;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private int maxRetries = Integer.MAX_VALUE;
        private long backoffSlotMillis = TimeUnit.SECONDS.toMillis(1L);
        private int backoffCeiling = 6; // Slot 1s, Max delay 32s

        private Builder() {
        }

        public Builder maxRetries(int maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public Builder backoffSlot(Duration duration) {
            checkArgument(!duration.isNegative(), "backoffSlot(%s) is negative", duration);
            this.backoffSlotMillis = duration.toMillis();
            return this;
        }

        public Builder backoffCeiling(int backoffCeiling) {
            this.backoffCeiling = backoffCeiling;
            return this;
        }

        public RetrySettings build() {
            return new RetrySettings(this);
        }
    }
}
