package ru.yandex.persqueue.read.settings;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class TopicReadSettings {
    public final String path;
    public final Instant startMessageTimestamp;
    public final List<Long> partitionGroupIds;

    public TopicReadSettings(Builder builder) {
        this.path = requireNonNull(Strings.emptyToNull(builder.path), "Topic path can't be empty");
        this.startMessageTimestamp = requireNonNull(builder.startMessageTimestamp);
        this.partitionGroupIds = List.copyOf(builder.partitionGroupIds);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        public String path;
        public Instant startMessageTimestamp = Instant.EPOCH;
        public List<Long> partitionGroupIds = new ArrayList<>();

        private Builder() {
        }

        /**
         * Path of topic to read
         */
        public Builder path(String path) {
            this.path = path;
            return this;
        }

        /**
         * Start reading from this timestamp
         */
        public Builder startMessageTimestamp(Instant timestamp) {
            this.startMessageTimestamp = timestamp;
            return this;
        }

        /**
         * Partition groups to read
         */
        public Builder partitionGroupIds(Long... groupIds) {
            this.partitionGroupIds.addAll(Arrays.asList(groupIds));
            return this;
        }

        public TopicReadSettings build() {
            return new TopicReadSettings(this);
        }
    }
}
