package ru.yandex.persqueue.rpc;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.WillClose;
import javax.annotation.WillNotClose;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.grpc.GrpcDiscoveryRpc;
import com.yandex.ydb.core.grpc.GrpcRequestSettings;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.OperationTray;
import com.yandex.ydb.core.rpc.OutStreamObserver;
import com.yandex.ydb.core.rpc.StreamObserver;
import com.yandex.ydb.discovery.DiscoveryProtos.ListEndpointsResult;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadClientMessage;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage;
import com.yandex.ydb.persqueue.cluster_discovery.YdbPersqueueClusterDiscovery.DiscoverClustersRequest;
import com.yandex.ydb.persqueue.cluster_discovery.YdbPersqueueClusterDiscovery.DiscoverClustersResult;
import com.yandex.ydb.persqueue.v1.ClusterDiscoveryServiceGrpc;
import io.netty.util.AbstractReferenceCounted;
import io.netty.util.ReferenceCounted;

import static com.yandex.ydb.persqueue.v1.PersQueueServiceGrpc.getMigrationStreamingReadMethod;


/**
 * @author Vladimir Gordiychuk
 */
public class GrpcPqRpc extends AbstractReferenceCounted implements PqRpc {
    @WillNotClose
    private final GrpcRpcPool pool;
    private final String endpoint;
    @WillClose
    private final GrpcTransport transport;
    private final AuthProvider authProvider;
    private final long readTimeoutNanos;
    private final OperationTray operationTray;

    public GrpcPqRpc(String endpoint, @WillNotClose GrpcRpcPool pool, @WillClose GrpcTransport transport, AuthProvider authProvider, Duration readTimeout) {
        this.endpoint = endpoint;
        this.pool = pool;
        this.transport = transport;
        this.authProvider = authProvider;
        this.operationTray = transport.getOperationTray();
        this.readTimeoutNanos = readTimeout.toNanos();
    }

    @Override
    public String nextToken() {
        return authProvider.getToken();
    }

    public String getEndpoint() {
        return endpoint;
    }

    @Override
    public CompletableFuture<Result<DiscoverClustersResult>> discoverClusters(DiscoverClustersRequest request) {
        return transport.unaryCall(ClusterDiscoveryServiceGrpc.getDiscoverClustersMethod(), request, deadlineAfter())
                .thenCompose(result -> {
                    if (!result.isSuccess()) {
                        return CompletableFuture.completedFuture(result.cast());
                    }

                    return operationTray.waitResult(
                            result.expect("discoverClusters()").getOperation(),
                            DiscoverClustersResult.class,
                            Function.identity(),
                            GrpcRequestSettings.newBuilder().withDeadlineAfter(deadlineAfter()).build());
                });
    }

    @Override
    public CompletableFuture<Result<ListEndpointsResult>> discoverNodes() {
        var rpc = new GrpcDiscoveryRpc(transport);
        return rpc.listEndpoints(transport.getDatabase(), GrpcRequestSettings.newBuilder().withDeadlineAfter(deadlineAfter()).build());
    }

    @Override
    public OutStreamObserver<MigrationStreamingReadClientMessage> readSessionV1(StreamObserver<MigrationStreamingReadServerMessage> observer) {
        // TODO: deal with no reply from server until long period of time
        return transport.bidirectionalStreamCall(getMigrationStreamingReadMethod(), observer, GrpcRequestSettings.newBuilder().build());
    }

    @Override
    public void close() {
        release();
    }

    @Override
    protected void deallocate() {
        pool.release(this);
        transport.close();
    }

    private long deadlineAfter() {
        return System.nanoTime() + readTimeoutNanos;
    }

    @Override
    public ReferenceCounted touch(Object hint) {
        return this;
    }
}
