package ru.yandex.persqueue.rpc;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.google.common.net.HostAndPort;
import com.yandex.ydb.core.grpc.GrpcTransport;
import io.netty.util.IllegalReferenceCountException;

import ru.yandex.persqueue.settings.GrpcTransportSettings;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcRpcPool implements RpcPool {
    private final GrpcTransportSettings settings;
    private final int defaultPort;
    private final ConcurrentMap<String, GrpcPqRpc> rpcByEndpoint = new ConcurrentHashMap<>();

    public GrpcRpcPool(GrpcTransportSettings settings) {
        this.settings = settings;
        this.defaultPort = HostAndPort.fromString(settings.endpoint).getPortOrDefault(GrpcTransport.DEFAULT_PORT);
    }

    @Override
    public PqRpc getRpc(String endpoint) {
        return rpcByEndpoint.compute(endpoint, (key, prev) -> {
            if (prev == null || prev.refCnt() == 0) {
                return makeRpc(endpoint);
            } else {
                try {
                    return (GrpcPqRpc) prev.retain();
                } catch (IllegalReferenceCountException e) {
                    return makeRpc(endpoint);
                }
            }
        });
    }

    void release(GrpcPqRpc rpc) {
        rpcByEndpoint.remove(rpc.getEndpoint(), rpc);
    }

    private GrpcPqRpc makeRpc(String endpoint) {
        var address = HostAndPort.fromString(endpoint);
        var transport = GrpcTransport.forHost(address.getHost(), address.getPortOrDefault(defaultPort))
                .withCallExecutor(settings.callExecutor)
                .withAuthProvider(settings.authProvider)
                .withDataBase(settings.database);

        if (settings.useTLS) {
            if (settings.cert.isEmpty()) {
                transport.withSecureConnection();
            } else {
                transport.withSecureConnection(settings.cert.toByteArray());
            }
        }
        transport.withChannelInitializer(settings.channelInitializer);
        return new GrpcPqRpc(endpoint, this, transport.build(), settings.authProvider, settings.readTimeout);
    }
}
