package ru.yandex.persqueue.settings;

import java.time.Duration;
import java.util.concurrent.Executor;
import java.util.function.Consumer;

import com.google.common.base.Strings;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.protobuf.ByteString;
import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.auth.NopAuthProvider;
import io.grpc.netty.NettyChannelBuilder;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcTransportSettings {
    public final Executor callExecutor;
    public final AuthProvider authProvider;
    public final Duration readTimeout;
    public final String endpoint;
    public final String database;
    public final ByteString cert;
    public final boolean useTLS;
    public final Consumer<NettyChannelBuilder> channelInitializer;

    private GrpcTransportSettings(Builder builder) {
        this.callExecutor = builder.callExecutor;
        this.authProvider = builder.authProvider;
        this.readTimeout = builder.readTimeout;
        this.endpoint = requireNonNull(Strings.nullToEmpty(builder.endpoint), "endpoint not specified");
        this.database = builder.database;
        this.cert = builder.cert;
        this.useTLS = builder.useTLS;
        this.channelInitializer = builder.channelInitializer;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private Executor callExecutor = MoreExecutors.directExecutor();
        private AuthProvider authProvider = NopAuthProvider.INSTANCE;
        private Duration readTimeout = Duration.ZERO;
        private String endpoint;
        private String database = "/Root";
        private ByteString cert = ByteString.EMPTY;
        private boolean useTLS = false;
        private Consumer<NettyChannelBuilder> channelInitializer = (cb) -> { };

        private Builder() {
        }

        public Builder callExecutor(Executor executor) {
            this.callExecutor = requireNonNull(executor, "callExecutor");
            return this;
        }

        public Builder authProvider(AuthProvider authProvider) {
            this.authProvider = requireNonNull(authProvider, "authProvider");
            return this;
        }

        public Builder readTimeout(Duration timeout) {
            this.readTimeout = timeout;
            checkArgument(timeout.toMillis() >= 0, "readTimeout must be greater than 0");
            return this;
        }

        public Builder endpoint(String endpoint) {
            this.endpoint = requireNonNull(Strings.emptyToNull(endpoint), "endpoint");
            return this;
        }

        public Builder database(String database) {
            this.database = requireNonNull(Strings.emptyToNull(database), "database");
            return this;
        }

        public Builder secureConnection(ByteString cert) {
            this.cert = cert;
            this.useTLS = true;
            return this;
        }

        public Builder secureConnection() {
            this.useTLS = true;
            return this;
        }

        public Builder channelInitializer(Consumer<NettyChannelBuilder> channelInitializer) {
            this.channelInitializer = channelInitializer;
            return this;
        }

        public GrpcTransportSettings build() {
            return new GrpcTransportSettings(this);
        }
    }

}
