package ru.yandex.persqueue.read;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;

import org.hamcrest.Matchers;

import ru.yandex.persqueue.read.event.Event;
import ru.yandex.persqueue.read.event.PartitionStreamCreateEvent;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class EventSubscriber implements Flow.Subscriber<Event> {
    public final ArrayBlockingQueue<Event> events = new ArrayBlockingQueue<>(10000);
    public final CompletableFuture<?> doneFuture = new CompletableFuture<>();
    public Flow.Subscription subscription;
    public volatile boolean autoRequestMore = true;

    @Override
    public void onSubscribe(Flow.Subscription subscription) {
        this.subscription = subscription;
        if (autoRequestMore) {
            subscription.request(1);
        }
    }

    @Override
    public void onNext(Event item) {
        events.add(item);
        if (autoRequestMore) {
            subscription.request(1);
        }
    }

    @Override
    public void onError(Throwable throwable) {
        doneFuture.completeExceptionally(throwable);
    }

    @Override
    public void onComplete() {
        doneFuture.complete(null);
    }

    public PartitionStreamCreateEvent expectAssign(long assignId, String cluster, String topic, long partition) {
        var event = takeEvent(PartitionStreamCreateEvent.class);
        var partitionStream = event.getPartitionStream();
        assertEquals(assignId, partitionStream.getAssignId());
        assertEquals(cluster, partitionStream.getClusterName());
        assertEquals(topic, partitionStream.getTopicPath());
        assertEquals(partition, partitionStream.getPartition());
        return event;
    }

    public <T> T takeEvent(Class<T> clazz) {
        Event event = takeEvent();
        assertThat(event, Matchers.instanceOf(clazz));
        return clazz.cast(event);
    }

    public Event takeEvent() {
        try {
            return events.take();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public void expectNoEvents() {
        assertEquals("events queue size on subscriber should be zero", 0, events.size());
    }
}
