package ru.yandex.persqueue.read.impl;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.LongStream;

import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.persqueue.YdbPersqueueV1;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.Assigned;
import org.hamcrest.Matchers;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.persqueue.read.PartitionStreamKey;
import ru.yandex.persqueue.read.impl.actor.ActorEvent;
import ru.yandex.persqueue.read.impl.actor.ActorEvents;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class PartitionStreamImplTest {
    @Rule
    public TestName name = new TestName();
    public final ArrayBlockingQueue<ActorEvent> events = new ArrayBlockingQueue<>(1000);

    @Test
    public void assign() {
        var assigment = prepareAssign();
        var partitionStream = createPartitionStream(assigment);

        partitionStream.confirmAssign(42, 10);
        var req = takeRequest(ActorEvents.ConfirmAssign.class);
        assertEquals(partitionStream, req.partitionStream);
        assertEquals(42, req.readOffset);
        assertEquals(10, req.commitOffset);
        expectNoEvents();
    }

    @Test
    public void destroy() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));

        partitionStream.confirmDestroy();
        var req = takeRequest(ActorEvents.ConfirmDestroy.class);
        assertEquals(partitionStream, req.partitionStream);
        expectNoEvents();
    }

    @Test
    public void partitionStatus() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));

        partitionStream.requestStatus();
        var req = takeRequest(ActorEvents.RequestPartitionStatus.class);
        assertEquals(partitionStream, req.partitionStream);
        expectNoEvents();
    }

    @Test(expected = UnexpectedResultException.class)
    public void unableToMultipleAssignConfirm() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);

        partitionStream.confirmAssign(0, 0);

        // double confirm assign
        partitionStream.confirmAssign(42, 10);
    }

    @Test(expected = UnexpectedResultException.class)
    public void unableToMultipleDestroy() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));

        partitionStream.confirmDestroy();
        assertNotNull(takeRequest(ActorEvents.ConfirmDestroy.class));

        // double confirm destroy
        partitionStream.confirmDestroy();
    }

    @Test
    public void concurrentAssignConfirmation() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        int success = IntStream.range(0, 100)
                .parallel()
                .map(value -> {
                    try {
                        partitionStream.confirmAssign(value, 0);
                        return 1;
                    } catch (Throwable e) {
                        return 0;
                    }
                })
                .sum();

        assertEquals("partition assign can be confirmed only once", 1, success);
        var req = takeRequest(ActorEvents.ConfirmAssign.class);
        assertEquals(partitionStream, req.partitionStream);
        expectNoEvents();
    }

    @Test
    public void concurrentDestroyConfirmation() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));

        int success = IntStream.range(0, 100)
                .parallel()
                .map(value -> {
                    try {
                        partitionStream.confirmDestroy();
                        return 1;
                    } catch (Throwable e) {
                        return 0;
                    }
                })
                .sum();

        assertEquals("partition destroy can be confirmed only once", 1, success);
        assertNotNull(takeRequest(ActorEvents.ConfirmDestroy.class));
        expectNoEvents();
    }

    @Test
    public void skipConfirmAssignAlreadyClosed() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.close();
        partitionStream.confirmAssign(0, 0);
        expectNoEvents();
    }

    @Test
    public void skipConfirmDestroyAlreadyClosed() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        var confirm = takeRequest(ActorEvents.ConfirmAssign.class);
        assertNotNull(confirm);
        partitionStream.close();
        partitionStream.confirmDestroy();
        expectNoEvents();
    }

    @Test
    public void skipPartitionStatusAlreadyClosed() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        var confirm = takeRequest(ActorEvents.ConfirmAssign.class);
        assertNotNull(confirm);
        partitionStream.close();
        partitionStream.requestStatus();
        expectNoEvents();
    }

    @Test(expected = UnexpectedResultException.class)
    public void commitUnreadOffset() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));
        partitionStream.commit(42);
    }

    @Test
    public void commit() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));
        partitionStream.addCookie(123, 42, 43);
        partitionStream.commit(42);

        var commit = takeRequest(ActorEvents.Commit.class);
        assertEquals(partitionStream, commit.partitionStream);
        assertEquals(123, commit.cookie.id);
        expectNoEvents();
    }

    @Test
    public void commitOnlyWhenAllRangeCommitted() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));
        partitionStream.addCookie(22, 42, 45);

        partitionStream.commit(42);
        expectNoEvents();
        partitionStream.commit(44);
        expectNoEvents();
        partitionStream.commit(43);

        var commit = takeRequest(ActorEvents.Commit.class);
        assertEquals(partitionStream, commit.partitionStream);
        assertEquals(22, commit.cookie.id);
        expectNoEvents();
    }

    @Test(expected = UnexpectedResultException.class)
    public void commitOffsetOnlyOnce() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));
        partitionStream.addCookie(22, 42, 45);

        partitionStream.commit(42);
        partitionStream.commit(42);
    }

    @Test
    public void concurrentCommitOneCookie() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));
        partitionStream.addCookie(42, 1, 100);

        {
            var offsets = LongStream.range(1, 100).boxed().collect(Collectors.toList());
            Collections.shuffle(offsets);
            offsets.parallelStream().forEach(partitionStream::commit);
        }

        var commit = takeRequest(ActorEvents.Commit.class);
        assertEquals(partitionStream, commit.partitionStream);
        assertEquals(42, commit.cookie.id);
        expectNoEvents();
    }

    @Test
    public void concurrentCommitManyCookie() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        assertNotNull(takeRequest(ActorEvents.ConfirmAssign.class));

        Set<Long> expectedCookie = new HashSet<>();
        for (int index = 1; index < 100; index++) {
            var cookie = ThreadLocalRandom.current().nextLong();
            expectedCookie.add(cookie);
            partitionStream.addCookie(cookie, index, index + 1);
        }

        {
            var offsets = LongStream.range(1, 100).boxed().collect(Collectors.toList());
            Collections.shuffle(offsets);
            offsets.parallelStream().forEach(partitionStream::commit);
        }

        while (!expectedCookie.isEmpty()) {
            var commit = takeRequest(ActorEvents.Commit.class);
            assertEquals(partitionStream, commit.partitionStream);
            assertTrue(expectedCookie.remove(commit.cookie.id));
        }

        expectNoEvents();
    }

    @Test
    public void skipCommitAlreadyClosed() {
        var assignment = prepareAssign();
        var partitionStream = createPartitionStream(assignment);
        partitionStream.confirmAssign(0, 0);
        var confirm = takeRequest(ActorEvents.ConfirmAssign.class);
        assertNotNull(confirm);
        partitionStream.addCookie(1, 1, 2);
        partitionStream.close();
        partitionStream.commit(1);
        expectNoEvents();
    }

    private void expectNoEvents() {
        assertEquals(0, events.size());
    }

    private <T> T takeRequest(Class<T> clazz) {
        var event = events.poll();
        assertNotNull(event);
        assertThat(event, Matchers.instanceOf(clazz));
        return clazz.cast(event);
    }

    private PartitionStreamImpl createPartitionStream(Assigned assigned) {
        var topic = assigned.getTopic().getPath();
        var cluster = assigned.getCluster();
        var partition = assigned.getPartition();
        var result = new PartitionStreamImpl(events::add, assigned);
        assertEquals(assigned.getAssignId(), result.getAssignId());
        assertEquals(cluster, result.getClusterName());
        assertEquals(topic, result.getTopicPath());
        assertEquals(partition, result.getPartition());
        assertEquals(new PartitionStreamKey(topic, cluster, partition), result.getKey());
        return result;
    }

    private Assigned prepareAssign() {
        return Assigned.newBuilder()
                .setAssignId(ThreadLocalRandom.current().nextLong())
                .setTopic(YdbPersqueueV1.Path.newBuilder().setPath("/topic/" + name.getMethodName()).build())
                .setCluster("cluster/" + name.getMethodName())
                .setPartition(ThreadLocalRandom.current().nextLong())
                .build();
    }
}
