package ru.yandex.persqueue.read.impl;

import java.util.Set;

import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.Assigned;
import com.yandex.ydb.persqueue.YdbPersqueueV1.Path;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.persqueue.read.PartitionStreamKey;
import ru.yandex.persqueue.read.impl.actor.ReadSessionActor;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class PartitionStreamMapTest {

    private final ReadSessionActor actor = event -> {};
    private PartitionStreamMap map;

    @Before
    public void setUp() {
        map = new PartitionStreamMap();
    }

    @Test
    public void empty() {
        assertNull(map.get(42));
        assertNull(map.get(new PartitionStreamKey("test", "test", 42)));;
        assertThat(map.partitionStreams(), Matchers.emptyIterable());
    }

    @Test
    public void add() {
        var one = partitionStream(1, "test", "alice", 42);
        var two = partitionStream(2, "test", "bob", 42);
        assertNull(map.add(one));
        assertNull(map.add(two));
        assertEquals(Set.of(one, two), Set.copyOf(map.partitionStreams()));
    }

    @Test
    public void get() {
        var alice = partitionStream(1, "test", "alice", 32);
        var bob = partitionStream(2, "test", "bob", 42);

        map.add(alice);
        map.add(bob);

        assertEquals(alice, map.get(alice.getAssignId()));
        assertEquals(alice, map.get(alice.getKey()));

        assertEquals(bob, map.get(bob.getAssignId()));
        assertEquals(bob, map.get(bob.getKey()));

        assertTrue(map.contains(alice));
        assertTrue(map.contains(bob));
    }

    @Test
    public void remove() {
        var alice = partitionStream(1, "test", "alice", 32);
        var bob = partitionStream(2, "test", "bob", 42);

        map.add(alice);
        map.add(bob);

        assertTrue(map.remove(alice));
        assertFalse(map.remove(alice));

        assertNull(map.get(alice.getAssignId()));
        assertNull(map.get(alice.getKey()));
        assertFalse(map.contains(alice));

        assertEquals(bob, map.get(bob.getAssignId()));
        assertEquals(bob, map.get(bob.getKey()));
        assertTrue(map.contains(bob));
        assertEquals(Set.of(bob), Set.copyOf(map.partitionStreams()));
    }

    @Test
    public void replace() {
        var one = partitionStream(1, "test", "alice", 42);
        var two = partitionStream(2, "test", "alice", 42);

        assertNull(map.add(one));
        assertEquals(one, map.add(two));

        assertTrue(map.contains(two));
        assertEquals(two, map.get(two.getAssignId()));
        assertEquals(two, map.get(two.getKey()));

        assertFalse(map.contains(one));
        assertNull(map.get(one.getAssignId()));
        assertEquals(two, map.get(one.getKey()));
        assertEquals(Set.of(two), Set.copyOf(map.partitionStreams()));
    }

    @Test
    public void clear() {
        var alice = partitionStream(1, "test", "alice", 32);
        var bob = partitionStream(2, "test", "bob", 42);

        map.add(alice);
        map.add(bob);

        map.clear();
        assertThat(map.partitionStreams(), Matchers.emptyIterable());

        assertNull(map.get(alice.getAssignId()));
        assertNull(map.get(alice.getKey()));
        assertFalse(map.contains(alice));

        assertNull(map.get(bob.getAssignId()));
        assertNull(map.get(bob.getKey()));
        assertFalse(map.contains(bob));
    }

    @Test(expected = UnexpectedResultException.class)
    public void assignIdUnique() {
        map.add(partitionStream(1, "test-1", "alice", 22));
        map.add(partitionStream(1, "test-2", "bob", 22));
    }

    private PartitionStreamImpl partitionStream(long assignId, String cluster, String path, long partition) {
        return new PartitionStreamImpl(actor, assign(assignId, cluster, path, partition));
    }

    private Assigned assign(long assignId, String cluster, String path, long partition) {
        return Assigned.newBuilder()
                .setTopic(Path.newBuilder().setPath(path))
                .setCluster(cluster)
                .setPartition(partition)
                .setAssignId(assignId)
                .build();
    }
}
