package ru.yandex.persqueue.read.impl;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.ByteString;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;

import ru.yandex.persqueue.read.EventSubscriber;
import ru.yandex.persqueue.read.event.Message;
import ru.yandex.persqueue.read.impl.actor.ServerSession;
import ru.yandex.persqueue.read.settings.EventHandlersSettings;
import ru.yandex.persqueue.read.settings.ReadSessionSettings;
import ru.yandex.persqueue.read.settings.RetrySettings;
import ru.yandex.persqueue.read.settings.TopicReadSettings;
import ru.yandex.persqueue.rpc.PqRpcStub;
import ru.yandex.persqueue.rpc.RpcPoolStub;

import static com.yandex.ydb.persqueue.YdbPersqueueV1.Codec.CODEC_RAW;
import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class ReadSessionImplTest {
    @Rule
    public TestName name = new TestName();
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private ReadSessionImpl session;
    private RpcPoolStub rpcPool;
    private PqRpcStub balancerRpc;
    private EventSubscriber subscriber;

    @Before
    public void setUp() {
        rpcPool = new RpcPoolStub();
        balancerRpc = rpcPool.getRpc(name.getMethodName());
        subscriber = new EventSubscriber();
        var settings = ReadSessionSettings.newBuilder()
                .addTopic(TopicReadSettings.newBuilder()
                        .path("/topic/alice")
                        .build())
                .addTopic(TopicReadSettings.newBuilder()
                        .path("/topic/bob")
                        .build())
                .consumerName(name.getMethodName() + "/consumer")
                .maxMemoryUsage(10 << 20)
                .executor(ForkJoinPool.commonPool())
                .eventHandlers(EventHandlersSettings.newBuilder()
                        .executor(ForkJoinPool.commonPool())
                        .commonHandler(subscriber)
                        .build())
                .retrySettings(RetrySettings.newBuilder()
                        .backoffSlot(Duration.ofMillis(5))
                        .build())
                .build();
        session = new ReadSessionImpl(balancerRpc.endpoint, rpcPool, settings);
    }

    @Test
    public void connectToEachCluster() {
        balancerRpc.endpointByCluster.put("man", "man.lbk.yandex-team.ru");
        balancerRpc.endpointByCluster.put("myt", "myt.lbk.yandex-team.ru");
        session.start();


        var man = serverSession("man", "man.lbk.yandex-team.ru");
        var myt = serverSession("myt", "myt.lbk.yandex-team.ru");

        // init send to both servers
        man.expectInit();
        myt.expectInit();

        man.sendInit();
        myt.sendInit();

        long manAssignId = man.expectAssign("alice", 1);
        long mytAssignId = myt.expectAssign("alice", 1);

        // receive man message
        {
            man.expectRead();
            var data = ByteString.copyFromUtf8("hi man");
            man.sendData(manAssignId, 42L, Map.of(), CODEC_RAW, ByteString.copyFromUtf8("hi man"));
            var event = subscriber.takeEvent(Message.class);
            assertEquals("man", event.getPartitionStream().getClusterName());
            assertEquals(manAssignId, event.getPartitionStream().getAssignId());
            assertEquals(data, event.getData());
        }

        // receive myt message
        {
            myt.expectRead();
            var data = ByteString.copyFromUtf8("hi myt");
            myt.sendData(mytAssignId, 42L, Map.of(), CODEC_RAW, ByteString.copyFromUtf8("hi myt"));
            var event = subscriber.takeEvent(Message.class);
            assertEquals("myt", event.getPartitionStream().getClusterName());
            assertEquals(mytAssignId, event.getPartitionStream().getAssignId());
            assertEquals(data, event.getData());
        }
    }

    @Test
    public void closeOneSessionCloseAnotherOne() {
        balancerRpc.endpointByCluster.put("man", "man.lbk.yandex-team.ru");
        balancerRpc.endpointByCluster.put("myt", "myt.lbk.yandex-team.ru");
        session.start();

        var man = serverSession("man", "man.lbk.yandex-team.ru");
        var myt = serverSession("myt", "myt.lbk.yandex-team.ru");

        // init send to both servers
        man.expectInit();
        myt.expectInit();

        man.sendError(Status.of(StatusCode.UNAUTHORIZED));
        man.expectComplete();
        myt.expectComplete();
    }

    @Test
    public void closeSessionsInitializedByUser() {
        balancerRpc.endpointByCluster.put("man", "man.lbk.yandex-team.ru");
        balancerRpc.endpointByCluster.put("myt", "myt.lbk.yandex-team.ru");
        session.start();

        var man = serverSession("man", "man.lbk.yandex-team.ru");
        var myt = serverSession("myt", "myt.lbk.yandex-team.ru");

        // init send to both servers
        man.expectInit();
        myt.expectInit();

        subscriber.subscription.cancel();
        man.expectComplete();
        myt.expectComplete();
    }

    public ServerSession serverSession(String cluster, String endpoint) {
        try {
            var rpc = rpcPool.getRpc(endpoint);
            while (true) {
                var observer = rpc.getActiveReadObserver();
                if (observer == null) {
                    TimeUnit.MILLISECONDS.sleep(10L);
                    continue;
                }

                return new ServerSession(observer, this.name.getMethodName(), cluster, subscriber);
            }
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }
}
