package ru.yandex.persqueue.read.impl.actor;

import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Flow;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Throwables;
import com.google.protobuf.ByteString;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.persqueue.YdbPersqueueV1;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.Assigned;
import com.yandex.ydb.persqueue.YdbPersqueueV1.MigrationStreamingReadServerMessage.DataBatch.MessageData;
import com.yandex.ydb.persqueue.YdbPersqueueV1.Path;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.persqueue.read.EventSubscriber;
import ru.yandex.persqueue.read.event.Event;
import ru.yandex.persqueue.read.event.Message;
import ru.yandex.persqueue.read.impl.PartitionStreamImpl;
import ru.yandex.persqueue.read.impl.actor.ActorEvents.MemoryChunkConsumed;
import ru.yandex.persqueue.read.impl.event.MessageImpl;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class EventPublisherTest {
    private static final int MEMORY_CHUNK_BYTES = 1 << 10; // 1 KiB

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(20, TimeUnit.SECONDS)
            .build();

    private ArrayBlockingQueue<ActorEvent> events;
    private PartitionStreamImpl partitionStream;
    private long offset;
    private long seqNo;

    @Before
    public void setUp() {
        events = new ArrayBlockingQueue<>(1000);
        partitionStream = new PartitionStreamImpl(events::add, prepareAssign());
        offset = 0;
        seqNo = 42;
    }

    @Test
    public void publisherClose() {
        EventSubscriber subscriber = new EventSubscriber();
        var publisher = newPublisher(subscriber);
        publisher.close();
        var error = subscriber.doneFuture
                .thenApply(ignore -> null)
                .exceptionally(e -> e)
                .join();
        assertNull(error);
    }

    @Test
    public void subscriberClose() throws InterruptedException {
        EventSubscriber subscriber = new EventSubscriber();
        try (var publisher = newPublisher(subscriber)) {
            subscriber.subscription.cancel();
            var event = events.take();
            assertThat(publisher.toString(), event, instanceOf(ActorEvents.Disconnect.class));
        }
    }

    @Test
    public void publisherError() {
        EventSubscriber subscriber = new EventSubscriber();
        try (var publisher = newPublisher(subscriber)) {
            publisher.closeExceptionally(Status.of(StatusCode.BAD_REQUEST));
            var error = subscriber.doneFuture
                    .thenApply(ignore -> null)
                    .exceptionally(e -> e)
                    .join();
            assertNotNull(error);
        }
    }

    @Test
    public void publisherSubmit() {
        EventSubscriber subscriber = new EventSubscriber();
        try (var publisher = newPublisher(subscriber)) {
            var expectedEvents = IntStream.range(0, 5000)
                    .mapToObj(ignore -> prepareNextMessage())
                    .collect(Collectors.toList());

            for (Event event : expectedEvents) {
                publisher.submit(event);
            }

            for (Event expected : expectedEvents) {
                var actual = subscriber.takeEvent(Message.class);
                assertEquals(expected, actual);
            }

            subscriber.expectNoEvents();
        }
    }

    @Test
    public void subscriberOnNextFailed() throws InterruptedException {
        var expectedError = new IllegalStateException("expected error");
        EventSubscriber subscriber = new EventSubscriber() {
            @Override
            public void onNext(Event item) {
                throw expectedError;
            }
        };

        try (var publisher = newPublisher(subscriber)) {
            publisher.submit(prepareNextMessage());
            var error = subscriber.doneFuture
                    .thenApply(ignore -> (Throwable) null)
                    .exceptionally(e -> e)
                    .join();

            assertNotNull(error);
            assertEquals(expectedError, Throwables.getRootCause(error));
            assertThat(events.take(), instanceOf(ActorEvents.Disconnect.class));
        }
    }

    @Test
    public void subscriberOnNextDemand() {
        EventSubscriber subscriber = new EventSubscriber();
        subscriber.autoRequestMore = false;

        try (var publisher = newPublisher(subscriber)) {
            var events = IntStream.range(0, 100)
                    .mapToObj(ignore -> prepareNextMessage())
                    .collect(Collectors.toList());

            events.forEach(publisher::submit);
            // requested 1
            {
                subscriber.subscription.request(1);
                var event = subscriber.takeEvent(Message.class);
                assertEquals(events.get(0), event);
                subscriber.expectNoEvents();
            }

            // requested 9
            {
                subscriber.subscription.request(9);
                for (int index = 1; index < 10; index++) {
                    var actual = subscriber.takeEvent(Message.class);
                    assertEquals(events.get(index), actual);
                }
                subscriber.expectNoEvents();
            }

            // requested by one
            {
                for (int index = 10; index < events.size(); index++) {
                    subscriber.subscription.request(1);
                    var actual = subscriber.takeEvent(Message.class);
                    assertEquals(events.get(index), actual);
                }
            }
        }
    }

    @Test
    public void memoryUseBytes() {
        EventSubscriber subscriber = new EventSubscriber();
        subscriber.autoRequestMore = false;

        try (var publisher = newPublisher(subscriber)) {
            var events = IntStream.range(0, 100)
                    .mapToObj(ignore -> prepareNextMessage())
                    .collect(Collectors.toList());

            assertEquals(0, publisher.getMemoryUseBytes("cluster/test"));
            int expectedSize = 0;
            for (var event : events) {
                expectedSize += event.memoryUseBytes();
                publisher.submit(event);
                assertEquals(expectedSize, publisher.getMemoryUseBytes("cluster/test"));
            }

            for (int index = 0; index < events.size(); index++) {
                subscriber.subscription.request(1);
                var event = subscriber.takeEvent(Message.class);
                expectedSize -= event.memoryUseBytes();
                assertEquals(expectedSize, publisher.getMemoryUseBytes("cluster/test"));
            }
            assertEquals(0, publisher.getMemoryUseBytes("cluster/test"));
        }
    }

    @Test
    public void sendEventEveryConsumedChunk() throws InterruptedException {
        EventSubscriber subscriber = new EventSubscriber();
        try (var publisher = newPublisher(subscriber)) {
            int consumedBytes = 0;

            while (consumedBytes < MEMORY_CHUNK_BYTES * 3) {
                var event = prepareNextMessage();
                consumedBytes += event.memoryUseBytes();
                publisher.submit(event);
                assertNotNull(subscriber.takeEvent(Message.class));
            }

            assertThat(events.take(), Matchers.instanceOf(ActorEvents.MemoryChunkConsumed.class));
            assertThat(events.take(), Matchers.instanceOf(ActorEvents.MemoryChunkConsumed.class));
            assertThat(events.take(), Matchers.instanceOf(ActorEvents.MemoryChunkConsumed.class));
        }
    }

    public EventPublisher newPublisher(Flow.Subscriber<? super Event> subscriber) {
        return new EventPublisher(new EventProducer() {
            @Override
            public void chunkConsumed(String cluster) {
                events.add(MemoryChunkConsumed.INSTANCE);
            }

            @Override
            public void cancel() {
                events.add(new ActorEvents.Disconnect());
            }
        }, ForkJoinPool.commonPool(), MEMORY_CHUNK_BYTES, subscriber);
    }

    private Assigned prepareAssign() {
        return Assigned.newBuilder()
                .setAssignId(ThreadLocalRandom.current().nextLong())
                .setTopic(Path.newBuilder().setPath("/topic/alice").build())
                .setCluster("cluster/test")
                .setPartition(ThreadLocalRandom.current().nextLong())
                .build();
    }

    private Message prepareNextMessage() {
        var data = MessageData.newBuilder()
                .setSeqNo(seqNo++)
                .setOffset(offset++)
                .setCodec(YdbPersqueueV1.Codec.CODEC_RAW)
                .setData(ByteString.copyFromUtf8("hi"))
                .build();
        return new MessageImpl(partitionStream, Map.of(), data);
    }
}
