package ru.yandex.persqueue.read.impl.actor;

import java.util.HashSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.protobuf.Duration;
import com.google.protobuf.Message;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class EventQueueImplTest {
    private EventQueue<Message> queue;

    @Before
    public void setUp() {
        queue = new EventQueueImpl<>(() -> {});
    }

    @Test
    public void emptyQueue() {
        assertNull(queue.dequeue());
        assertNull(queue.getError());
        assertFalse(queue.isCompleted());
    }

    @Test
    public void enqueueDequeue() {
        var message = randomMessage();
        queue.enqueue(message);

        assertFalse(queue.isCompleted());
        assertNull(queue.getError());
        assertEquals(message, queue.dequeue());
        assertNull(queue.dequeue());
    }

    @Test
    public void sameOrder() {
        var expected = IntStream.range(0, 1000)
                .mapToObj(ignore -> randomMessage())
                .collect(Collectors.toList());

        expected.forEach(queue::enqueue);
        assertNull(queue.getError());
        assertFalse(queue.isCompleted());

        for (Message message : expected) {
            assertEquals(message, queue.dequeue());
        }

        assertNull(queue.dequeue());
    }

    @Test
    public void concurrentEnqueue() {
        var expected = IntStream.range(0, 1000)
                .mapToObj(ignore -> randomMessage())
                .collect(Collectors.toList());

        expected.parallelStream().forEach(queue::enqueue);

        var expectedSet = new HashSet<>(expected);
        while (!expectedSet.isEmpty()) {
            var message = queue.dequeue();
            assertNotNull(message);
            assertTrue(expectedSet.remove(message));
        }
    }

    @Test
    public void error() {
        var expectedStatus = Status.of(StatusCode.BAD_SESSION);
        queue.enqueue(randomMessage());
        queue.onError(expectedStatus);

        assertNotNull(queue.dequeue());
        assertEquals(expectedStatus, queue.getError());
    }

    @Test
    public void complete() {
        queue.enqueue(randomMessage());
        queue.onComplete();

        assertNotNull(queue.dequeue());
        assertTrue(queue.isCompleted());
    }

    @Test
    public void onEnqueueCallback() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        queue = new EventQueueImpl<>(sync::countDown);

        ForkJoinPool.commonPool().execute(() -> queue.enqueue(randomMessage()));
        assertTrue(sync.await(1, TimeUnit.SECONDS));
    }

    @Test
    public void onErrorCallback() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        queue = new EventQueueImpl<>(sync::countDown);

        ForkJoinPool.commonPool().execute(() -> queue.onError(Status.of(StatusCode.ABORTED)));
        assertTrue(sync.await(1, TimeUnit.SECONDS));
    }

    @Test
    public void onCompleteCallback() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        queue = new EventQueueImpl<>(sync::countDown);

        ForkJoinPool.commonPool().execute(() -> queue.onComplete());
        assertTrue(sync.await(1, TimeUnit.SECONDS));
    }

    @Test
    public void onErrorIgnoreOtherEvents() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        queue = new EventQueueImpl<>(sync::countDown);
        assertFalse(queue.isDone());

        var error = Status.of(StatusCode.BAD_REQUEST);
        queue.onError(error);
        queue.onComplete();
        queue.enqueue(randomMessage());

        sync.await();
        assertTrue(queue.isDone());
        assertFalse(queue.isCompleted());
        assertEquals(error, queue.getError());
        assertNull(queue.dequeue());
    }

    @Test
    public void onCompleteIgnoreOtherEvents() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        queue = new EventQueueImpl<>(sync::countDown);
        assertFalse(queue.isDone());

        queue.onComplete();
        queue.onError(Status.of(StatusCode.BAD_REQUEST));
        queue.enqueue(randomMessage());

        sync.await();
        assertTrue(queue.isDone());
        assertTrue(queue.isCompleted());
        assertNull(queue.getError());
        assertNull(queue.dequeue());
    }

    private Message randomMessage() {
        var random = ThreadLocalRandom.current();
        return Duration.newBuilder()
                .setSeconds(random.nextLong())
                .setNanos(random.nextInt())
                .build();
    }
}
